/*
 * Copyright (c) 2011 Vojtech Horky
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * - Redistributions of source code must retain the above copyright
 *   notice, this list of conditions and the following disclaimer.
 * - Redistributions in binary form must reproduce the above copyright
 *   notice, this list of conditions and the following disclaimer in the
 *   documentation and/or other materials provided with the distribution.
 * - The name of the author may not be used to endorse or promote products
 *   derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/** @addtogroup drvusbmid
 * @{
 */
/**
 * @file
 * Exploration of available interfaces in the USB device.
 */
#include <errno.h>
#include <str_error.h>
#include <stdlib.h>
#include <usb/classes/classes.h>
#include <usb/dev/request.h>
#include <usb/dev/dp.h>
#include <usb/ddfiface.h>
#include "usbmid.h"

/** Operations of the device itself. */
static ddf_dev_ops_t mid_device_ops = {
	.interfaces[USB_DEV_IFACE] = &usb_iface_hub_impl
};

/** Tell whether given interface is already in the list.
 *
 * @param list List of usbmid_interface_t members to be searched.
 * @param interface_no Interface number caller is looking for.
 * @return Interface @p interface_no is already present in the list.
 */
static bool interface_in_list(link_t *list, int interface_no)
{
	link_t *l;
	for (l = list->next; l != list; l = l->next) {
		usbmid_interface_t *iface
		    = list_get_instance(l, usbmid_interface_t, link);
		if (iface->interface_no == interface_no) {
			return true;
		}
	}

	return false;
}

/** Create list of interfaces from configuration descriptor.
 *
 * @param config_descriptor Configuration descriptor.
 * @param config_descriptor_size Size of configuration descriptor in bytes.
 * @param list List where to add the interfaces.
 */
static void create_interfaces(uint8_t *config_descriptor,
    size_t config_descriptor_size, link_t *list)
{
	usb_dp_parser_data_t data = {
		.data = config_descriptor,
		.size = config_descriptor_size,
		.arg = NULL
	};

	usb_dp_parser_t parser = {
		.nesting = usb_dp_standard_descriptor_nesting
	};

	uint8_t *interface_ptr = usb_dp_get_nested_descriptor(&parser, &data,
	    data.data);
	if (interface_ptr == NULL) {
		return;
	}

	do {
		if (interface_ptr[1] != USB_DESCTYPE_INTERFACE) {
			goto next_descriptor;
		}

		usb_standard_interface_descriptor_t *interface
		    = (usb_standard_interface_descriptor_t *) interface_ptr;

		/* Skip alternate interfaces. */
		if (!interface_in_list(list, interface->interface_number)) {
			usbmid_interface_t *iface
			    = malloc(sizeof(usbmid_interface_t));
			if (iface == NULL) {
				break;
			}
			link_initialize(&iface->link);
			iface->fun = NULL;
			iface->interface_no = interface->interface_number;
			iface->interface = interface;

			list_append(&iface->link, list);
		}

		/* TODO: add the alternatives and create match ids from them
		 * as well.
		 */

next_descriptor:
		interface_ptr = usb_dp_get_sibling_descriptor(&parser, &data,
		    data.data, interface_ptr);

	} while (interface_ptr != NULL);

}

/** Explore MID device.
 *
 * We expect that @p dev is initialized and session on control pipe is
 * started.
 *
 * @param dev Device to be explored.
 * @return Whether to accept this device from devman.
 */
bool usbmid_explore_device(usb_device_t *dev)
{
	int rc;

	int dev_class = dev->descriptors.device.device_class;
	if (dev_class != USB_CLASS_USE_INTERFACE) {
		usb_log_warning(
		    "Device class: %d (%s), but expected class 0.\n",
		    dev_class, usb_str_class(dev_class));
		usb_log_error("Not multi interface device, refusing.\n");
		return false;
	}

	/* Short cuts to save on typing ;-). */
	uint8_t *config_descriptor_raw = dev->descriptors.configuration;
	size_t config_descriptor_size = dev->descriptors.configuration_size;
	usb_standard_configuration_descriptor_t *config_descriptor =
	    (usb_standard_configuration_descriptor_t *) config_descriptor_raw;

	/* Select the first configuration */
	rc = usb_request_set_configuration(&dev->ctrl_pipe,
	    config_descriptor->configuration_number);
	if (rc != EOK) {
		usb_log_error("Failed to set device configuration: %s.\n",
		    str_error(rc));
		return false;
	}

	/* Create control function */
	ddf_fun_t *ctl_fun = ddf_fun_create(dev->ddf_dev, fun_exposed, "ctl");
	if (ctl_fun == NULL) {
		usb_log_error("Failed to create control function.\n");
		return false;
	}

	ctl_fun->ops = &mid_device_ops;

	rc = ddf_fun_bind(ctl_fun);
	if (rc != EOK) {
		usb_log_error("Failed to bind control function: %s.\n",
		    str_error(rc));
		return false;
	}

	/* Create interface children. */
	link_t interface_list;
	list_initialize(&interface_list);
	create_interfaces(config_descriptor_raw, config_descriptor_size,
	    &interface_list);

	link_t *link;
	for (link = interface_list.next; link != &interface_list;
	    link = link->next) {
		usbmid_interface_t *iface = list_get_instance(link,
		    usbmid_interface_t, link);

		usb_log_info("Creating child for interface %d (%s).\n",
		    (int) iface->interface_no,
		    usb_str_class(iface->interface->interface_class));

		rc = usbmid_spawn_interface_child(dev, iface,
		    &dev->descriptors.device, iface->interface);
		if (rc != EOK) {
			usb_log_error("Failed to create interface child: %s.\n",
			    str_error(rc));
		}
	}

	return true;
}

/**
 * @}
 */
