#
# Copyright (C) 2005 Jakub Jermar
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions
# are met:
#
# - Redistributions of source code must retain the above copyright
#   notice, this list of conditions and the following disclaimer.
# - Redistributions in binary form must reproduce the above copyright
#   notice, this list of conditions and the following disclaimer in the
#   documentation and/or other materials provided with the distribution.
# - The name of the author may not be used to endorse or promote products
#   derived from this software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
# IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
# OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
# IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
# INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
# NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
# DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
# THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
# (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
# THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#

/**
 * This file contains kernel trap table.
 */

.register %g2, #scratch
.register %g3, #scratch

.text

#include <arch/trap/trap_table.h>
#include <arch/trap/regwin.h>
#include <arch/trap/interrupt.h>
#include <arch/trap/exception.h>
#include <arch/trap/mmu.h>
#include <arch/stack.h>
#include <arch/regdef.h>

#define TABLE_SIZE	TRAP_TABLE_SIZE
#define ENTRY_SIZE	TRAP_TABLE_ENTRY_SIZE

/*
 * Kernel trap table.
 */
.align TABLE_SIZE
.global trap_table
trap_table:

/* TT = 0x08, TL = 0, instruction_access_exception */
.org trap_table + TT_INSTRUCTION_ACCESS_EXCEPTION*ENTRY_SIZE
.global instruction_access_exception
instruction_access_exception:
	SIMPLE_HANDLER do_instruction_access_exc

/* TT = 0x10, TL = 0, illegal_instruction */
.org trap_table + TT_ILLEGAL_INSTRUCTION*ENTRY_SIZE
.global illegal_instruction
illegal_instruction:
	SIMPLE_HANDLER do_illegal_instruction

/* TT = 0x24, TL = 0, clean_window handler */
.org trap_table + TT_CLEAN_WINDOW*ENTRY_SIZE
.global clean_window_handler
clean_window_handler:
	CLEAN_WINDOW_HANDLER

/* TT = 0x32, TL = 0, data_access_error */
.org trap_table + TT_DATA_ACCESS_ERROR*ENTRY_SIZE
.global data_access_error
data_access_error:
	SIMPLE_HANDLER do_data_access_error

/* TT = 0x34, TL = 0, mem_address_not_aligned */
.org trap_table + TT_MEM_ADDRESS_NOT_ALIGNED*ENTRY_SIZE
.global mem_address_not_aligned
mem_address_not_aligned:
	SIMPLE_HANDLER do_mem_address_not_aligned

/* TT = 0x41, TL = 0, interrupt_level_1 handler */
.org trap_table + TT_INTERRUPT_LEVEL_1*ENTRY_SIZE
.global interrupt_level_1_handler
interrupt_level_1_handler:
	INTERRUPT_LEVEL_N_HANDLER 1

/* TT = 0x42, TL = 0, interrupt_level_2 handler */
.org trap_table + TT_INTERRUPT_LEVEL_2*ENTRY_SIZE
.global interrupt_level_2_handler
interrupt_level_2_handler:
	INTERRUPT_LEVEL_N_HANDLER 2

/* TT = 0x43, TL = 0, interrupt_level_3 handler */
.org trap_table + TT_INTERRUPT_LEVEL_3*ENTRY_SIZE
.global interrupt_level_3_handler
interrupt_level_3_handler:
	INTERRUPT_LEVEL_N_HANDLER 3

/* TT = 0x44, TL = 0, interrupt_level_4 handler */
.org trap_table + TT_INTERRUPT_LEVEL_4*ENTRY_SIZE
.global interrupt_level_4_handler
interrupt_level_4_handler:
	INTERRUPT_LEVEL_N_HANDLER 4

/* TT = 0x45, TL = 0, interrupt_level_5 handler */
.org trap_table + TT_INTERRUPT_LEVEL_5*ENTRY_SIZE
.global interrupt_level_5_handler
interrupt_level_5_handler:
	INTERRUPT_LEVEL_N_HANDLER 5

/* TT = 0x46, TL = 0, interrupt_level_6 handler */
.org trap_table + TT_INTERRUPT_LEVEL_6*ENTRY_SIZE
.global interrupt_level_6_handler
interrupt_level_6_handler:
	INTERRUPT_LEVEL_N_HANDLER 6

/* TT = 0x47, TL = 0, interrupt_level_7 handler */
.org trap_table + TT_INTERRUPT_LEVEL_7*ENTRY_SIZE
.global interrupt_level_7_handler
interrupt_level_7_handler:
	INTERRUPT_LEVEL_N_HANDLER 7

/* TT = 0x48, TL = 0, interrupt_level_8 handler */
.org trap_table + TT_INTERRUPT_LEVEL_8*ENTRY_SIZE
.global interrupt_level_8_handler
interrupt_level_8_handler:
	INTERRUPT_LEVEL_N_HANDLER 8

/* TT = 0x49, TL = 0, interrupt_level_9 handler */
.org trap_table + TT_INTERRUPT_LEVEL_9*ENTRY_SIZE
.global interrupt_level_9_handler
interrupt_level_9_handler:
	INTERRUPT_LEVEL_N_HANDLER 9

/* TT = 0x4a, TL = 0, interrupt_level_10 handler */
.org trap_table + TT_INTERRUPT_LEVEL_10*ENTRY_SIZE
.global interrupt_level_10_handler
interrupt_level_10_handler:
	INTERRUPT_LEVEL_N_HANDLER 10

/* TT = 0x4b, TL = 0, interrupt_level_11 handler */
.org trap_table + TT_INTERRUPT_LEVEL_11*ENTRY_SIZE
.global interrupt_level_11_handler
interrupt_level_11_handler:
	INTERRUPT_LEVEL_N_HANDLER 11

/* TT = 0x4c, TL = 0, interrupt_level_12 handler */
.org trap_table + TT_INTERRUPT_LEVEL_12*ENTRY_SIZE
.global interrupt_level_12_handler
interrupt_level_12_handler:
	INTERRUPT_LEVEL_N_HANDLER 12

/* TT = 0x4d, TL = 0, interrupt_level_13 handler */
.org trap_table + TT_INTERRUPT_LEVEL_13*ENTRY_SIZE
.global interrupt_level_13_handler
interrupt_level_13_handler:
	INTERRUPT_LEVEL_N_HANDLER 13

/* TT = 0x4e, TL = 0, interrupt_level_14 handler */
.org trap_table + TT_INTERRUPT_LEVEL_14*ENTRY_SIZE
.global interrupt_level_14_handler
interrupt_level_14_handler:
	INTERRUPT_LEVEL_N_HANDLER 14

/* TT = 0x4f, TL = 0, interrupt_level_15 handler */
.org trap_table + TT_INTERRUPT_LEVEL_15*ENTRY_SIZE
.global interrupt_level_15_handler
interrupt_level_15_handler:
	INTERRUPT_LEVEL_N_HANDLER 15

/* TT = 0x60, TL = 0, interrupt_vector_trap handler */
.org trap_table + TT_INTERRUPT_VECTOR_TRAP*ENTRY_SIZE
.global interrupt_vector_trap_handler
interrupt_vector_trap_handler:
	INTERRUPT_VECTOR_TRAP_HANDLER

/* TT = 0x64, TL = 0, fast_instruction_access_MMU_miss */
.org trap_table + TT_FAST_INSTRUCTION_ACCESS_MMU_MISS*ENTRY_SIZE
.global fast_instruction_access_mmu_miss_handler
fast_instruction_access_mmu_miss_handler:
	FAST_INSTRUCTION_ACCESS_MMU_MISS_HANDLER

/* TT = 0x68, TL = 0, fast_data_access_MMU_miss */
.org trap_table + TT_FAST_DATA_ACCESS_MMU_MISS*ENTRY_SIZE
.global fast_data_access_mmu_miss_handler
fast_data_access_mmu_miss_handler:
	FAST_DATA_ACCESS_MMU_MISS_HANDLER

/* TT = 0x6c, TL = 0, fast_data_access_protection */
.org trap_table + TT_FAST_DATA_ACCESS_PROTECTION*ENTRY_SIZE
.global fast_data_access_protection_handler
fast_data_access_protection_handler:
	FAST_DATA_ACCESS_PROTECTION_HANDLER

/* TT = 0x80, TL = 0, spill_0_normal handler */
.org trap_table + TT_SPILL_0_NORMAL*ENTRY_SIZE
.global spill_0_normal
spill_0_normal:
	SPILL_NORMAL_HANDLER_KERNEL

/* TT = 0xc0, TL = 0, fill_0_normal handler */
.org trap_table + TT_FILL_0_NORMAL*ENTRY_SIZE
.global fill_0_normal
fill_0_normal:
	FILL_NORMAL_HANDLER_KERNEL

/*
 * Handlers for TL>0.
 */

/* TT = 0x08, TL > 0, instruction_access_exception */
.org trap_table + (TT_INSTRUCTION_ACCESS_EXCEPTION+512)*ENTRY_SIZE
.global instruction_access_exception_high
instruction_access_exception_high:
	SIMPLE_HANDLER do_instruction_access_exc

/* TT = 0x10, TL > 0, illegal_instruction */
.org trap_table + (TT_ILLEGAL_INSTRUCTION+512)*ENTRY_SIZE
.global illegal_instruction_high
illegal_instruction_high:
	SIMPLE_HANDLER do_illegal_instruction

/* TT = 0x24, TL > 0, clean_window handler */
.org trap_table + (TT_CLEAN_WINDOW+512)*ENTRY_SIZE
.global clean_window_handler_high
clean_window_handler_high:
	CLEAN_WINDOW_HANDLER

/* TT = 0x32, TL > 0, data_access_error */
.org trap_table + (TT_DATA_ACCESS_ERROR+512)*ENTRY_SIZE
.global data_access_error_high
data_access_error_high:
	SIMPLE_HANDLER do_data_access_error

/* TT = 0x34, TL > 0, mem_address_not_aligned */
.org trap_table + (TT_MEM_ADDRESS_NOT_ALIGNED+512)*ENTRY_SIZE
.global mem_address_not_aligned_high
mem_address_not_aligned_high:
	SIMPLE_HANDLER do_mem_address_not_aligned

/* TT = 0x64, TL > 0, fast_instruction_access_MMU_miss */
.org trap_table + (TT_FAST_INSTRUCTION_ACCESS_MMU_MISS+512)*ENTRY_SIZE
.global fast_instruction_access_mmu_miss_handler_high
fast_instruction_access_mmu_miss_handler_high:
	FAST_INSTRUCTION_ACCESS_MMU_MISS_HANDLER

/* TT = 0x68, TL > 0, fast_data_access_MMU_miss */
.org trap_table + (TT_FAST_DATA_ACCESS_MMU_MISS+512)*ENTRY_SIZE
.global fast_data_access_mmu_miss_handler_high
fast_data_access_mmu_miss_handler_high:
	FAST_DATA_ACCESS_MMU_MISS_HANDLER

/* TT = 0x6c, TL > 0, fast_data_access_protection */
.org trap_table + (TT_FAST_DATA_ACCESS_PROTECTION+512)*ENTRY_SIZE
.global fast_data_access_protection_handler_high
fast_data_access_protection_handler_high:
	FAST_DATA_ACCESS_PROTECTION_HANDLER

/* TT = 0x80, TL > 0, spill_0_normal handler */
.org trap_table + (TT_SPILL_0_NORMAL+512)*ENTRY_SIZE
.global spill_0_normal_high
spill_0_normal_high:
	SPILL_NORMAL_HANDLER_KERNEL

/* TT = 0xc0, TL > 0, fill_0_normal handler */
.org trap_table + (TT_FILL_0_NORMAL+512)*ENTRY_SIZE
.global fill_0_normal_high
fill_0_normal_high:
	FILL_NORMAL_HANDLER_KERNEL


/* Preemptible trap handler for TL=1.
 *
 * This trap handler makes arrangements to make calling of scheduler() from
 * within a trap context possible. It is called from several other trap
 * handlers.
 *
 * This function can be entered either with interrupt globals or alternate globals.
 * Memory management trap handlers are obliged to switch to one of those global sets
 * prior to calling this function. Register window management functions are not
 * allowed to modify the alternate global registers.
 *
 * Input registers:
 *	%g1		Address of function to call.
 * 	%g2	 	Argument for the function.
 *	%g6		Pre-set as kernel stack base if trap from userspace.
 *	%g7		Reserved.
 */
.global preemptible_handler
preemptible_handler:
	rdpr %tstate, %g3
	andcc %g3, TSTATE_PRIV_BIT, %g0		! if this trap came from the privileged mode...
	bnz 0f					! ...skip setting of kernel stack and primary context
	nop

	/*
	 * Switch to kernel stack. The old stack is
	 * automatically saved in the old window's %sp
	 * and the new window's %fp.
	 */
	save %g6, -PREEMPTIBLE_HANDLER_STACK_FRAME_SIZE, %sp

	/*
	 * Mark the CANSAVE windows as OTHER windows.
	 * Set CLEANWIN to NWINDOW-1 so that clean_window traps do not occur.
	 */
	rdpr %cansave, %l0
	wrpr %l0, %otherwin
	wrpr %g0, %cansave
	wrpr %g0, NWINDOW-1, %cleanwin

	/*
	 * Switch to primary context 0.
	 */
	mov VA_PRIMARY_CONTEXT_REG, %l0
        stxa %g0, [%l0] ASI_DMMU
	set kernel_image_start, %l0
        flush %l0

	ba 1f
	nop

0:
	save %sp, -PREEMPTIBLE_HANDLER_STACK_FRAME_SIZE, %sp

	/*
	 * At this moment, we are using the kernel stack 
	 * and have successfully allocated a register window.
	 */
1:

	/*
	 * Copy arguments.
	 */
	mov %g1, %l0
	mov %g2, %o0

	/*
	 * Save TSTATE, TPC and TNPC aside.
	 */
	rdpr %tstate, %g1
	rdpr %tpc, %g2
	rdpr %tnpc, %g3

	/*
	 * The following memory accesses will not fault
	 * because special provisions are made to have
	 * the kernel stack of THREAD locked in DTLB.
	 */
	stx %g1, [%sp + PREEMPTIBLE_HANDLER_STACK_FRAME_SIZE + STACK_BIAS + SAVED_TSTATE]
	stx %g2, [%sp + PREEMPTIBLE_HANDLER_STACK_FRAME_SIZE + STACK_BIAS + SAVED_TPC]
	stx %g3, [%sp + PREEMPTIBLE_HANDLER_STACK_FRAME_SIZE + STACK_BIAS + SAVED_TNPC]
	
	wrpr %g0, 0, %tl
	wrpr %g0, PSTATE_PRIV_BIT, %pstate
	SAVE_GLOBALS
	
	/*
	 * Call the higher-level handler and pass istate as second parameter.
	 */
	call %l0
	add %sp, PREEMPTIBLE_HANDLER_STACK_FRAME_SIZE + STACK_BIAS + SAVED_TNPC, %o1

	RESTORE_GLOBALS
	wrpr %g0, PSTATE_AG_BIT | PSTATE_PRIV_BIT, %pstate
	wrpr %g0, 1, %tl
	
	/*
	 * Read TSTATE, TPC and TNPC from saved copy.
	 */
	ldx [%sp + PREEMPTIBLE_HANDLER_STACK_FRAME_SIZE + STACK_BIAS + SAVED_TSTATE], %g1
	ldx [%sp + PREEMPTIBLE_HANDLER_STACK_FRAME_SIZE + STACK_BIAS + SAVED_TPC], %g2
	ldx [%sp + PREEMPTIBLE_HANDLER_STACK_FRAME_SIZE + STACK_BIAS + SAVED_TNPC], %g3

	/*
	 * Restore TSTATE, TPC and TNPC from saved copies.
	 */
	wrpr %g1, 0, %tstate
	wrpr %g2, 0, %tpc
	wrpr %g3, 0, %tnpc

	/*
	 * If OTHERWIN is zero, then all the userspace windows have been
	 * spilled to kernel memory (i.e. register window buffer). If
	 * OTHERWIN is non-zero, then some userspace windows are still
	 * valid. Others might have been spilled. However, the CWP pointer
	 * needs no fixing because the scheduler had not been called.
	 */
	rdpr %otherwin, %l0
	brnz %l0, 0f
	nop

	/*
	 * OTHERWIN == 0
	 */

	/*
	 * If TSTATE.CWP + 1 == CWP, then we still do not have to fix CWP.
	 */
	and %g1, TSTATE_CWP_MASK, %l0
	inc %l0
	and %l0, TSTATE_CWP_MASK, %l0	! %l0 mod NWINDOW
	rdpr %cwp, %l1
	cmp %l0, %l1
	bz 0f				! CWP is ok
	nop

	/*
	 * Fix CWP.
	 */
	mov %fp, %g1
	flushw
	wrpr %l0, 0, %cwp
	mov %g1, %fp
	
	/*
	 * OTHERWIN != 0 or fall-through from the OTHERWIN == 0 case.
	 */
0:
	! TODO: restore register windows from register window memory buffer

	restore
	retry
