/*
 * Copyright (c) 2011 Jiri Svoboda
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * - Redistributions of source code must retain the above copyright
 *   notice, this list of conditions and the following disclaimer.
 * - Redistributions in binary form must reproduce the above copyright
 *   notice, this list of conditions and the following disclaimer in the
 *   documentation and/or other materials provided with the distribution.
 * - The name of the author may not be used to endorse or promote products
 *   derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/**
 * @addtogroup libnic
 * @{
 */
/**
 * @file
 * @brief
 */

#include <async.h>
#include <device/nic.h>
#include "nic_ev.h"

/** Device address changed. */
int nic_ev_addr_changed(async_sess_t *sess, nic_device_id_t dev_id,
    const nic_address_t *addr)
{
	async_exch_t *exch = async_exchange_begin(sess);

	ipc_call_t answer;
	aid_t req = async_send_1(exch, NIC_EV_ADDR_CHANGED, (sysarg_t) dev_id,
	    &answer);
	sysarg_t retval = async_data_write_start(exch, addr,
	    sizeof(nic_address_t));

	async_exchange_end(exch);

	if (retval != EOK) {
		async_wait_for(req, NULL);
		return retval;
	}

	async_wait_for(req, &retval);
	return retval;
}

/** Device state changed. */
extern int nic_ev_device_state(async_sess_t *sess, nic_device_id_t dev_id,
    sysarg_t state)
{
	int rc;

	async_exch_t *exch = async_exchange_begin(sess);
	rc = async_req_2_0(exch, NIC_EV_DEVICE_STATE, dev_id, state);
	async_exchange_end(exch);

	return rc;
}

/** Frame received. */
int nic_ev_received(async_sess_t *sess, nic_device_id_t dev_id, void *data,
    size_t size)
{
	async_exch_t *exch = async_exchange_begin(sess);

	ipc_call_t answer;
	aid_t req = async_send_1(exch, NIC_EV_RECEIVED, (sysarg_t) dev_id,
	    &answer);
	sysarg_t retval = async_data_write_start(exch, data, size);

	async_exchange_end(exch);

	if (retval != EOK) {
		async_wait_for(req, NULL);
		return retval;
	}

	async_wait_for(req, &retval);
	return retval;
}

/** @}
 */
