/*
 * Copyright (c) 2009 Martin Decky
 * Copyright (c) 2010 Jiri Svoboda
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * - Redistributions of source code must retain the above copyright
 *   notice, this list of conditions and the following disclaimer.
 * - Redistributions in binary form must reproduce the above copyright
 *   notice, this list of conditions and the following disclaimer in the
 *   documentation and/or other materials provided with the distribution.
 * - The name of the author may not be used to endorse or promote products
 *   derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/** @addtogroup genarch
 * @{
 */
/**
 * @file
 * @brief Samsung S3C24xx on-chip UART driver.
 *
 * This UART is present on the Samsung S3C24xx CPU (on the gta02 platform).
 */

#include <genarch/drivers/s3c24xx_uart/s3c24xx_uart.h>
#include <console/chardev.h>
#include <console/console.h>
#include <ddi/device.h>
#include <arch/asm.h>
#include <mm/slab.h>
#include <sysinfo/sysinfo.h>
#include <str.h>

/* Bits in UTRSTAT register */
#define S3C24XX_UTRSTAT_TX_EMPTY	0x4
#define S3C24XX_UTRSTAT_RDATA		0x1

static void s3c24xx_uart_sendb(outdev_t *dev, uint8_t byte)
{
	s3c24xx_uart_t *uart =
	    (s3c24xx_uart_t *) dev->data;

	/* Wait for transmitter to be empty. */
	while ((pio_read_32(&uart->io->utrstat) & S3C24XX_UTRSTAT_TX_EMPTY) == 0)
		;

	pio_write_32(&uart->io->utxh, byte);
}

static void s3c24xx_uart_putchar(outdev_t *dev, wchar_t ch, bool silent)
{
	if (!silent) {
		if (!ascii_check(ch)) {
			s3c24xx_uart_sendb(dev, U_SPECIAL);
		} else {
    			if (ch == '\n')
				s3c24xx_uart_sendb(dev, (uint8_t) '\r');
			s3c24xx_uart_sendb(dev, (uint8_t) ch);
		}
	}
}

static irq_ownership_t s3c24xx_uart_claim(irq_t *irq)
{
	return IRQ_ACCEPT;
}

static void s3c24xx_uart_irq_handler(irq_t *irq)
{
	s3c24xx_uart_t *uart = irq->instance;

	if ((pio_read_32(&uart->io->utrstat) & S3C24XX_UTRSTAT_RDATA) != 0) {
		uint32_t data = pio_read_32(&uart->io->urxh);
		indev_push_character(uart->indev, data & 0xff);
	}
}

static outdev_operations_t s3c24xx_uart_ops = {
	.write = s3c24xx_uart_putchar,
	.redraw = NULL
};

outdev_t *s3c24xx_uart_init(s3c24xx_uart_io_t *io, inr_t inr)
{
	outdev_t *uart_dev = malloc(sizeof(outdev_t), FRAME_ATOMIC);
	if (!uart_dev)
		return NULL;

	s3c24xx_uart_t *uart =
	    malloc(sizeof(s3c24xx_uart_t), FRAME_ATOMIC);
	if (!uart) {
		free(uart_dev);
		return NULL;
	}

	outdev_initialize("s3c24xx_uart_dev", uart_dev, &s3c24xx_uart_ops);
	uart_dev->data = uart;

	uart->io = io;
	uart->indev = NULL;

	/* Initialize IRQ structure. */
	irq_initialize(&uart->irq);
	uart->irq.devno = device_assign_devno();
	uart->irq.inr = inr;
	uart->irq.claim = s3c24xx_uart_claim;
	uart->irq.handler = s3c24xx_uart_irq_handler;
	uart->irq.instance = uart;

	/* Disable FIFO */
	pio_write_32(&uart->io->ufcon,
	    pio_read_32(&uart->io->ufcon) & ~0x01);

	/* Set RX interrupt to pulse mode */
	pio_write_32(&uart->io->ucon,
	    pio_read_32(&uart->io->ucon) & ~(1 << 8));

	if (!fb_exported) {
		/*
		 * This is the necessary evil until the userspace driver is entirely
		 * self-sufficient.
		 */
		sysinfo_set_item_val("fb", NULL, true);
		sysinfo_set_item_val("fb.kind", NULL, 3);
		sysinfo_set_item_val("fb.address.physical", NULL, KA2PA(io));

		fb_exported = true;
	}

	return uart_dev;
}

void s3c24xx_uart_input_wire(s3c24xx_uart_t *uart, indev_t *indev)
{
	ASSERT(uart);
	ASSERT(indev);

	uart->indev = indev;
	irq_register(&uart->irq);
}

/** @}
 */
