/*
 * Copyright (c) 2010 Lenka Trochtova
 * Copyright (c) 2015 Jakub Jermar
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * - Redistributions of source code must retain the above copyright
 *   notice, this list of conditions and the following disclaimer.
 * - Redistributions in binary form must reproduce the above copyright
 *   notice, this list of conditions and the following disclaimer in the
 *   documentation and/or other materials provided with the distribution.
 * - The name of the author may not be used to endorse or promote products
 *   derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/**
 * @defgroup msim MSIM platform driver.
 * @brief HelenOS MSIM platform driver.
 * @{
 */

/** @file
 */

#include <assert.h>
#include <stdio.h>
#include <errno.h>
#include <stdbool.h>
#include <stdlib.h>
#include <ctype.h>
#include <macros.h>

#include <ddi.h>
#include <ddf/driver.h>
#include <ddf/log.h>
#include <ipc/dev_iface.h>
#include <ops/hw_res.h>
#include <ops/pio_window.h>

#define NAME "msim"

#define MSIM_DISK_BASE	UINT32_C(0x10000200)
#define MSIM_DISK_SIZE	UINT32_C(0x00000010)

typedef struct msim_fun {
	hw_resource_list_t hw_resources;
	pio_window_t pio_window;
} msim_fun_t;

static int msim_dev_add(ddf_dev_t *dev);
static void msim_init(void);

/** The root device driver's standard operations. */
static driver_ops_t msim_ops = {
	.dev_add = &msim_dev_add
};

/** The root device driver structure. */
static driver_t msim_driver = {
	.name = NAME,
	.driver_ops = &msim_ops
};

static hw_resource_t disk_regs[] = {
	{
		.type = MEM_RANGE,
		.res.mem_range = {
			.address = 0,
			.size = 16,
			.relative = true,
			.endianness = LITTLE_ENDIAN
		}
	},
	{
		.type = INTERRUPT,
		.res.interrupt = {
			.irq = 6
		}
	}
};

static msim_fun_t disk_data = {
	.hw_resources = {
		sizeof(disk_regs) / sizeof(disk_regs[0]),
		disk_regs
	},
	.pio_window = {
		.mem = {
			.base = MSIM_DISK_BASE,
			.size = MSIM_DISK_SIZE
		}
	}
};

/** Obtain function soft-state from DDF function node */
static msim_fun_t *msim_fun(ddf_fun_t *fnode)
{
	return ddf_fun_data_get(fnode);
}

static hw_resource_list_t *msim_get_resources(ddf_fun_t *fnode)
{
	msim_fun_t *fun = msim_fun(fnode);
	
	assert(fun != NULL);
	return &fun->hw_resources;
}

static bool msim_enable_interrupt(ddf_fun_t *fun)
{
	/* Nothing to do. */

	return true;
}

static pio_window_t *msim_get_pio_window(ddf_fun_t *fnode)
{
	msim_fun_t *fun = msim_fun(fnode);

	assert(fun != NULL);
	return &fun->pio_window;
}

static hw_res_ops_t fun_hw_res_ops = {
	.get_resource_list = &msim_get_resources,
	.enable_interrupt = &msim_enable_interrupt,
};

static pio_window_ops_t fun_pio_window_ops = {
	.get_pio_window = &msim_get_pio_window
};

/* Initialized in msim_init() function. */
static ddf_dev_ops_t msim_fun_ops;

static bool
msim_add_fun(ddf_dev_t *dev, const char *name, const char *str_match_id,
    msim_fun_t *fun_proto)
{
	ddf_msg(LVL_DEBUG, "Adding new function '%s'.", name);
	
	ddf_fun_t *fnode = NULL;
	int rc;
	
	/* Create new device. */
	fnode = ddf_fun_create(dev, fun_inner, name);
	if (fnode == NULL)
		goto failure;
	
	msim_fun_t *fun = ddf_fun_data_alloc(fnode, sizeof(msim_fun_t));
	*fun = *fun_proto;
	
	/* Add match ID */
	rc = ddf_fun_add_match_id(fnode, str_match_id, 100);
	if (rc != EOK)
		goto failure;
	
	/* Set provided operations to the device. */
	ddf_fun_set_ops(fnode, &msim_fun_ops);
	
	/* Register function. */
	if (ddf_fun_bind(fnode) != EOK) {
		ddf_msg(LVL_ERROR, "Failed binding function %s.", name);
		goto failure;
	}
	
	return true;
	
failure:
	if (fnode != NULL)
		ddf_fun_destroy(fnode);
	
	ddf_msg(LVL_ERROR, "Failed adding function '%s'.", name);
	
	return false;
}

static bool msim_add_functions(ddf_dev_t *dev)
{
	return msim_add_fun(dev, "disk0", "msim/ddisk", &disk_data);
}

/** Get the root device.
 *
 * @param dev		The device which is root of the whole device tree (both
 *			of HW and pseudo devices).
 * @return		Zero on success, negative error number otherwise.
 */
static int msim_dev_add(ddf_dev_t *dev)
{
	ddf_msg(LVL_DEBUG, "msim_dev_add, device handle = %d",
	    (int) ddf_dev_get_handle(dev));

	/* Register functions. */
	if (!msim_add_functions(dev))
		ddf_msg(LVL_ERROR, "Failed to add functions for the MSIM platform.");
	
	return EOK;
}

static void msim_init(void)
{
	ddf_log_init(NAME);
	msim_fun_ops.interfaces[HW_RES_DEV_IFACE] = &fun_hw_res_ops;
	msim_fun_ops.interfaces[PIO_WINDOW_DEV_IFACE] = &fun_pio_window_ops;
}

int main(int argc, char *argv[])
{
	printf(NAME ": HelenOS MSIM platform driver\n");
	msim_init();
	return ddf_driver_main(&msim_driver);
}

/**
 * @}
 */
