/*
 * Copyright (c) 2008 Jakub Jermar 
 * Copyright (c) 2008 Martin Decky 
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * - Redistributions of source code must retain the above copyright
 *   notice, this list of conditions and the following disclaimer.
 * - Redistributions in binary form must reproduce the above copyright
 *   notice, this list of conditions and the following disclaimer in the
 *   documentation and/or other materials provided with the distribution.
 * - The name of the author may not be used to endorse or promote products
 *   derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/** @addtogroup fs
 * @{
 */ 

/**
 * @file	tmpfs_dump.c
 * @brief	Support for loading TMPFS file system dump.
 */

#include "tmpfs.h"
#include "../../vfs/vfs.h"
#include <ipc/ipc.h>
#include <async.h>
#include <errno.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <as.h>
#include <libblock.h>
#include <ipc/services.h>
#include <ipc/devmap.h>
#include <sys/mman.h>
#include <byteorder.h>

#define TMPFS_BLOCK_SIZE	1024

struct rdentry {
	uint8_t type;
	uint32_t len;
} __attribute__((packed));

static bool
tmpfs_restore_recursion(int phone, void *block, off_t *bufpos, size_t *buflen,
    off_t *pos, tmpfs_dentry_t *parent)
{
	struct rdentry entry;
	libfs_ops_t *ops = &tmpfs_libfs_ops;
	
	do {
		char *fname;
		tmpfs_dentry_t *node;
		uint32_t size;
		
		if (!blockread(phone, block, bufpos, buflen, pos, &entry,
		    sizeof(entry), TMPFS_BLOCK_SIZE))
			return false;
		
		entry.len = uint32_t_le2host(entry.len);
		
		switch (entry.type) {
		case TMPFS_NONE:
			break;
		case TMPFS_FILE:
			fname = malloc(entry.len + 1);
			if (fname == NULL)
				return false;
			
			node = (tmpfs_dentry_t *) ops->create(L_FILE);
			if (node == NULL) {
				free(fname);
				return false;
			}
			
			if (!blockread(phone, block, bufpos, buflen, pos, fname,
			    entry.len, TMPFS_BLOCK_SIZE)) {
				ops->destroy((void *) node);
				free(fname);
				return false;
			}
			fname[entry.len] = 0;
			
			if (!ops->link((void *) parent, (void *) node, fname)) {
				ops->destroy((void *) node);
				free(fname);
				return false;
			}
			free(fname);
			
			if (!blockread(phone, block, bufpos, buflen, pos, &size,
			    sizeof(size), TMPFS_BLOCK_SIZE))
				return false;
			
			size = uint32_t_le2host(size);
			
			node->data = malloc(size);
			if (node->data == NULL)
				return false;
			
			node->size = size;
			if (!blockread(phone, block, bufpos, buflen, pos,
			    node->data, size, TMPFS_BLOCK_SIZE))
				return false;
			
			break;
		case TMPFS_DIRECTORY:
			fname = malloc(entry.len + 1);
			if (fname == NULL)
				return false;
			
			node = (tmpfs_dentry_t *) ops->create(L_DIRECTORY);
			if (node == NULL) {
				free(fname);
				return false;
			}
			
			if (!blockread(phone, block, bufpos, buflen, pos, fname,
			    entry.len, TMPFS_BLOCK_SIZE)) {
				ops->destroy((void *) node);
				free(fname);
				return false;
			}
			fname[entry.len] = 0;
			
			if (!ops->link((void *) parent, (void *) node, fname)) {
				ops->destroy((void *) node);
				free(fname);
				return false;
			}
			free(fname);
			
			if (!tmpfs_restore_recursion(phone, block, bufpos,
			    buflen, pos, node))
				return false;
			
			break;
		default:
			return false;
		}
	} while (entry.type != TMPFS_NONE);
	
	return true;
}

bool tmpfs_restore(dev_handle_t dev)
{
	libfs_ops_t *ops = &tmpfs_libfs_ops;

	void *block = mmap(NULL, TMPFS_BLOCK_SIZE,
	    PROTO_READ | PROTO_WRITE, MAP_ANONYMOUS | MAP_PRIVATE, 0, 0);
	
	if (block == NULL)
		return false;
	
	int phone = ipc_connect_me_to(PHONE_NS, SERVICE_DEVMAP,
	    DEVMAP_CONNECT_TO_DEVICE, dev);

	if (phone < 0) {
		munmap(block, TMPFS_BLOCK_SIZE);
		return false;
	}
	
	if (ipc_share_out_start(phone, block, AS_AREA_READ | AS_AREA_WRITE) !=
	    EOK)
		goto error;
	
	off_t bufpos = 0;
	size_t buflen = 0;
	off_t pos = 0;
	
	char tag[6];
	if (!blockread(phone, block, &bufpos, &buflen, &pos, tag, 5,
	    TMPFS_BLOCK_SIZE))
		goto error;
	
	tag[5] = 0;
	if (strcmp(tag, "TMPFS") != 0)
		goto error;
	
	if (!tmpfs_restore_recursion(phone, block, &bufpos, &buflen, &pos,
	    ops->root_get(dev)))
		goto error;
		
	ipc_hangup(phone);
	munmap(block, TMPFS_BLOCK_SIZE);
	return true;
	
error:
	ipc_hangup(phone);
	munmap(block, TMPFS_BLOCK_SIZE);
	return false;
}

/**
 * @}
 */ 
