/*
 * Copyright (c) 2024 Miroslav Cimerman
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * - Redistributions of source code must retain the above copyright
 *   notice, this list of conditions and the following disclaimer.
 * - Redistributions in binary form must reproduce the above copyright
 *   notice, this list of conditions and the following disclaimer in the
 *   documentation and/or other materials provided with the distribution.
 * - The name of the author may not be used to endorse or promote products
 *   derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/** @addtogroup hr
 * @{
 */
/**
 * @file
 */

#ifndef _HR_VAR_H
#define _HR_VAR_H

#include <bd_srv.h>
#include <errno.h>
#include <hr.h>

#define NAME "hr"

#define HR_STRIP_SIZE DATA_XFER_LIMIT

typedef struct hr_volume hr_volume_t;

typedef struct hr_ops {
	errno_t (*create)(hr_volume_t *);
	errno_t (*init)(hr_volume_t *);
	void	(*status_event)(hr_volume_t *);
	errno_t	(*add_hotspare)(hr_volume_t *, service_id_t);
} hr_ops_t;

typedef struct hr_volume {
	hr_ops_t hr_ops;
	bd_srvs_t hr_bds;

	link_t lvolumes;
	fibril_mutex_t lock;

	size_t extent_no;
	hr_extent_t extents[HR_MAX_EXTENTS];

	size_t hotspare_no;
	hr_extent_t hotspares[HR_MAX_EXTENTS];

	size_t bsize;
	uint64_t nblocks;
	uint64_t data_blkno;
	uint64_t data_offset; /* in blocks */
	uint32_t strip_size;

	uint64_t rebuild_blk;

	uint64_t counter; /* metadata syncing */

	service_id_t svc_id;
	hr_vol_status_t status;
	hr_level_t level;
	uint8_t RLQ; /* RAID Level Qualifier */
	char devname[HR_DEVNAME_LEN];
} hr_volume_t;

typedef enum {
	HR_BD_SYNC,
	HR_BD_READ,
	HR_BD_WRITE
} hr_bd_op_type_t;

extern errno_t hr_init_devs(hr_volume_t *);
extern void hr_fini_devs(hr_volume_t *);

extern errno_t hr_raid0_create(hr_volume_t *);
extern errno_t hr_raid1_create(hr_volume_t *);
extern errno_t hr_raid5_create(hr_volume_t *);

extern errno_t hr_raid0_init(hr_volume_t *);
extern errno_t hr_raid1_init(hr_volume_t *);
extern errno_t hr_raid5_init(hr_volume_t *);

extern void hr_raid0_status_event(hr_volume_t *);
extern void hr_raid1_status_event(hr_volume_t *);
extern void hr_raid5_status_event(hr_volume_t *);

extern errno_t hr_raid1_add_hotspare(hr_volume_t *, service_id_t);
extern errno_t hr_raid5_add_hotspare(hr_volume_t *, service_id_t);

#endif

/** @}
 */
