/*
 * Copyright (C) 2006 Ondrej Palkovsky
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * - Redistributions of source code must retain the above copyright
 *   notice, this list of conditions and the following disclaimer.
 * - Redistributions in binary form must reproduce the above copyright
 *   notice, this list of conditions and the following disclaimer in the
 *   documentation and/or other materials provided with the distribution.
 * - The name of the author may not be used to endorse or promote products
 *   derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/* Lock ordering
 *
 * First the answerbox, then the phone
 */

#include <synch/spinlock.h>
#include <synch/waitq.h>
#include <ipc/ipc.h>
#include <errno.h>
#include <mm/slab.h>
#include <arch.h>
#include <proc/task.h>
#include <memstr.h>
#include <debug.h>

#include <print.h>
#include <proc/thread.h>

/* Open channel that is assigned automatically to new tasks */
answerbox_t *ipc_phone_0 = NULL;

static slab_cache_t *ipc_call_slab;

/** Allocate & initialize call structure
 * 
 * The call is initialized, so that the reply will be directed
 * to TASK->answerbox
 */
call_t * ipc_call_alloc(void)
{
	call_t *call;

	call = slab_alloc(ipc_call_slab, 0);
	memsetb((__address)call, sizeof(*call), 0);
	call->callerbox = &TASK->answerbox;
	call->sender = TASK;

	return call;
}

/** Initialize allocated call */
void ipc_call_init(call_t *call)
{
	call->callerbox = &TASK->answerbox;
	call->flags = IPC_CALL_STATIC_ALLOC;
	call->sender = TASK;
}

/** Deallocate call stracuture */
void ipc_call_free(call_t *call)
{
	slab_free(ipc_call_slab, call);
}

/** Initialize answerbox structure
 */
void ipc_answerbox_init(answerbox_t *box)
{
	spinlock_initialize(&box->lock, "ipc_box_lock");
	waitq_initialize(&box->wq);
	list_initialize(&box->connected_phones);
	list_initialize(&box->calls);
	list_initialize(&box->dispatched_calls);
	list_initialize(&box->answers);
	box->task = TASK;
}

/** Connect phone to answerbox */
void ipc_phone_connect(phone_t *phone, answerbox_t *box)
{
	ASSERT(!phone->callee);
	phone->busy = 1;
	phone->callee = box;

	spinlock_lock(&box->lock);
	list_append(&phone->list, &box->connected_phones);
	spinlock_unlock(&box->lock);
}

/** Initialize phone structure and connect phone to naswerbox
 */
void ipc_phone_init(phone_t *phone)
{
	spinlock_initialize(&phone->lock, "phone_lock");
	phone->callee = NULL;
	phone->busy = 0;
}

/** Disconnect phone from answerbox */
void ipc_phone_destroy(phone_t *phone)
{
	answerbox_t *box = phone->callee;
	
	ASSERT(box);

	spinlock_lock(&box->lock);
	list_remove(&phone->list);
	spinlock_unlock(&box->lock);
}

/** Helper function to facilitate synchronous calls */
void ipc_call_sync(phone_t *phone, call_t *request)
{
	answerbox_t sync_box; 

	ipc_answerbox_init(&sync_box);

	/* We will receive data on special box */
	request->callerbox = &sync_box;

	ipc_call(phone, request);
	ipc_wait_for_call(&sync_box, 0);
}

/** Send a asynchronous request using phone to answerbox
 *
 * @param phone Phone connected to answerbox
 * @param request Request to be sent
 */
void ipc_call(phone_t *phone, call_t *call)
{
	answerbox_t *box = phone->callee;

	ASSERT(box);

	spinlock_lock(&box->lock);
	list_append(&call->list, &box->calls);
	spinlock_unlock(&box->lock);
	waitq_wakeup(&box->wq, 0);
}

/** Forwards call from one answerbox to a new one
 *
 * @param request Request to be forwarded
 * @param newbox Target answerbox
 * @param oldbox Old answerbox
 */
void ipc_forward(call_t *call, answerbox_t *newbox, answerbox_t *oldbox)
{
	spinlock_lock(&oldbox->lock);
	list_remove(&call->list);
	spinlock_unlock(&oldbox->lock);

	spinlock_lock(&newbox->lock);
	list_append(&call->list, &newbox->calls);
	spinlock_lock(&newbox->lock);
	waitq_wakeup(&newbox->wq, 0);
}

/** Answer message back to phone
 *
 * @param box Answerbox that is answering the message
 * @param request Modified request that is being sent back
 */
void ipc_answer(answerbox_t *box, call_t *request)
{
	answerbox_t *callerbox = request->callerbox;

	request->flags &= ~IPC_CALL_DISPATCHED;
	request->flags |= IPC_CALL_ANSWERED;

	spinlock_lock(&box->lock);
	list_remove(&request->list);
	spinlock_unlock(&box->lock);

	spinlock_lock(&callerbox->lock);
	list_append(&request->list, &callerbox->answers);
	spinlock_unlock(&callerbox->lock);
	waitq_wakeup(&callerbox->wq, 0);
}

/** Wait for phone call 
 *
 * @return Recived message address
 * - to distinguish between call and answer, look at call->flags
 */
call_t * ipc_wait_for_call(answerbox_t *box, int flags)
{
	call_t *request;

	spinlock_lock(&box->lock);
	while (1) { 
		if (!list_empty(&box->answers)) {
			/* Handle asynchronous answers */
			request = list_get_instance(box->answers.next, call_t, list);
			list_remove(&request->list);
		} else if (!list_empty(&box->calls)) {
			/* Handle requests */
			request = list_get_instance(box->calls.next, call_t, list);
			list_remove(&request->list);
			/* Append request to dispatch queue */
			list_append(&request->list, &box->dispatched_calls);
			request->flags |= IPC_CALL_DISPATCHED;
		} else {
			if (!(flags & IPC_WAIT_NONBLOCKING)) {
				/* Wait for event to appear */
				spinlock_unlock(&box->lock);
				waitq_sleep(&box->wq);
				spinlock_lock(&box->lock);
				continue;
			}
			request = NULL;
		}
		break;
	}
	spinlock_unlock(&box->lock);
	return request;
}

/** Initilize ipc subsystem */
void ipc_init(void)
{
	ipc_call_slab = slab_cache_create("ipc_call",
					  sizeof(call_t),
					  0,
					  NULL, NULL, 0);
}
