/*
 * Copyright (c) 2012 Jan Vesely
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * - Redistributions of source code must retain the above copyright
 *   notice, this list of conditions and the following disclaimer.
 * - Redistributions in binary form must reproduce the above copyright
 *   notice, this list of conditions and the following disclaimer in the
 *   documentation and/or other materials provided with the distribution.
 * - The name of the author may not be used to endorse or promote products
 *   derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
/** @addtogroup arm32beagleboardxm
 * @{
 */
/** @file
 *  @brief BeagleBoard-xM platform driver.
 */

#include <arch/exception.h>
#include <arch/mach/beagleboardxm/beagleboardxm.h>
#include <genarch/drivers/amdm37x_irc/amdm37x_irc.h>
#include <genarch/drivers/amdm37x_uart/amdm37x_uart.h>
#include <interrupt.h>
#include <mm/km.h>
#include <ddi/ddi.h>
#include <ddi/device.h>

static void bbxm_init(void);
static void bbxm_timer_irq_start(void);
static void bbxm_cpu_halt(void);
static void bbxm_get_memory_extents(uintptr_t *start, size_t *size);
static void bbxm_irq_exception(unsigned int exc_no, istate_t *istate);
static void bbxm_frame_init(void);
static void bbxm_output_init(void);
static void bbxm_input_init(void);
static size_t bbxm_get_irq_count(void);
static const char *bbxm_get_platform_name(void);

static struct beagleboard {
	amdm37x_irc_regs_t *irc_addr;
	amdm37x_uart_t uart;
} beagleboard;

struct arm_machine_ops bbxm_machine_ops = {
	bbxm_init,
	bbxm_timer_irq_start,
	bbxm_cpu_halt,
	bbxm_get_memory_extents,
	bbxm_irq_exception,
	bbxm_frame_init,
	bbxm_output_init,
	bbxm_input_init,
	bbxm_get_irq_count,
	bbxm_get_platform_name
};

static irq_ownership_t bb_timer_irq_claim(irq_t *irq)
{
	return IRQ_ACCEPT;
}

static void bb_timer_irq_handler(irq_t *irq)
{
        /*
         * We are holding a lock which prevents preemption.
         * Release the lock, call clock() and reacquire the lock again.
         */
        spinlock_unlock(&irq->lock);
        clock();
        spinlock_lock(&irq->lock);
}

static void bbxm_init(void)
{
	/* Initialize interrupt controller */
	beagleboard.irc_addr =
	    (void *) km_map(AMDM37x_IRC_BASE_ADDRESS, AMDM37x_IRC_SIZE,
	    PAGE_NOT_CACHEABLE);
	amdm37x_irc_init(beagleboard.irc_addr);

	//initialize timer here
}

static void bbxm_timer_irq_start(void)
{
	/* Initialize timer IRQ */
	static irq_t timer_irq;
	irq_initialize(&timer_irq);
	timer_irq.devno = device_assign_devno();
	timer_irq.inr = 0;//BB_TIMER_IRQ;
	timer_irq.claim = bb_timer_irq_claim;
	timer_irq.handler = bb_timer_irq_handler;
	irq_register(&timer_irq);
	// start timer here
}

static void bbxm_cpu_halt(void)
{
}

/** Get extents of available memory.
 *
 * @param start		Place to store memory start address (physical).
 * @param size		Place to store memory size.
 */
static void bbxm_get_memory_extents(uintptr_t *start, size_t *size)
{
	// FIXME: This is just a guess...
	*start = 0;
	*size = 256 * 1024 * 1024;
}

static void bbxm_irq_exception(unsigned int exc_no, istate_t *istate)
{
	const unsigned inum = amdm37x_irc_inum_get(beagleboard.irc_addr);
	amdm37x_irc_irq_ack(beagleboard.irc_addr);

	irq_t *irq = irq_dispatch_and_lock(inum);
	if (irq) {
		/* The IRQ handler was found. */
		irq->handler(irq);
		spinlock_unlock(&irq->lock);
	} else {
		/* Spurious interrupt.*/
		printf("cpu%d: spurious interrupt (inum=%d)\n",
		    CPU->id, inum);
	}
}

static void bbxm_frame_init(void)
{
}

static void bbxm_output_init(void)
{
#ifdef CONFIG_FB
#error "Frame buffer is not yet supported!"
#endif

	/* UART3 is wired to external RS232 connector */
	const bool ok = amdm37x_uart_init(&beagleboard.uart,
	    AMDM37x_UART3_IRQ, AMDM37x_UART3_BASE_ADDRESS, AMDM37x_UART3_SIZE);
	if (ok) {
		stdout_wire(&beagleboard.uart.outdev);
	}
}

static void bbxm_input_init(void)
{
}

size_t bbxm_get_irq_count(void)
{
	return AMDM37x_IRC_IRQ_COUNT;
}

const char *bbxm_get_platform_name(void)
{
	return "beagleboardxm";
}

/**
 * @}
 */
