/*
 * Copyright (c) 2011 Martin Sucha
 * Copyright (c) 2010 Jiri Svoboda
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * - Redistributions of source code must retain the above copyright
 *   notice, this list of conditions and the following disclaimer.
 * - Redistributions in binary form must reproduce the above copyright
 *   notice, this list of conditions and the following disclaimer in the
 *   documentation and/or other materials provided with the distribution.
 * - The name of the author may not be used to endorse or promote products
 *   derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/** @addtogroup fs
 * @{
 */

/**
 * @file	ext2info.c
 * @brief	Tool for displaying information about ext2 filesystem
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <libblock.h>
#include <mem.h>
#include <devmap.h>
#include <byteorder.h>
#include <sys/types.h>
#include <sys/typefmt.h>
#include <inttypes.h>
#include <errno.h>
#include <libext2.h>

#define NAME	"ext2info"

static void syntax_print(void);
static void print_superblock(ext2_superblock_t *);
static void print_block_groups(ext2_filesystem_t *);
static void print_block_group(ext2_block_group_t *);
static void print_inode_by_number(ext2_filesystem_t *, uint32_t);
static void print_inode(ext2_inode_t *);

#define ARG_SUPERBLOCK 1
#define ARG_BLOCK_GROUPS 2
#define ARG_INODE 4
#define ARG_STRICT_CHECK 8
#define ARG_COMMON (ARG_SUPERBLOCK | ARG_BLOCK_GROUPS)
#define ARG_ALL (ARG_COMMON | ARG_INODE)


int main(int argc, char **argv)
{

	int rc;
	char *endptr;
	char *dev_path;
	devmap_handle_t handle;
	ext2_filesystem_t filesystem;
	int arg_flags;
	uint32_t inode = 0;
	
	arg_flags = 0;
	
	if (argc < 2) {
		printf(NAME ": Error, argument missing.\n");
		syntax_print();
		return 1;
	}
	
	// Skip program name
	--argc; ++argv;
	
	if (str_cmp(*argv, "--strict-check") == 0) {
		--argc; ++argv;
		arg_flags |= ARG_STRICT_CHECK;
	}
	
	if (str_cmp(*argv, "--superblock") == 0) {
		--argc; ++argv;
		arg_flags |= ARG_SUPERBLOCK;
	}
	
	if (str_cmp(*argv, "--block-groups") == 0) {
		--argc; ++argv;
		arg_flags |= ARG_BLOCK_GROUPS;
	}
	
	if (str_cmp(*argv, "--inode") == 0) {
		--argc; ++argv;
		if (argc == 0) {
			printf(NAME ": Argument expected for --inode\n");
			return 2;
		}
		
		inode = strtol(*argv, &endptr, 10);
		if (*endptr != '\0') {
			printf(NAME ": Error, invalid argument for --inode.\n");
			syntax_print();
			return 1;
		}
		
		arg_flags |= ARG_INODE;
		--argc; ++argv;
	}

	if (argc != 1) {
		printf(NAME ": Error, unexpected argument.\n");
		syntax_print();
		return 1;
	}
	
	// Display common things by default
	if ((arg_flags & ARG_ALL) == 0) {
		arg_flags = ARG_COMMON;
	}

	dev_path = *argv;

	rc = devmap_device_get_handle(dev_path, &handle, 0);
	if (rc != EOK) {
		printf(NAME ": Error resolving device `%s'.\n", dev_path);
		return 2;
	}

	rc = ext2_filesystem_init(&filesystem, handle);
	if (rc != EOK)  {
		printf(NAME ": Error initializing libext2.\n");
		return 3;
	}
	
	rc = ext2_filesystem_check_sanity(&filesystem);
	if (rc != EOK) {
		printf(NAME ": Filesystem did not pass sanity check.\n");
		if (arg_flags & ARG_STRICT_CHECK) {
			return 3;
		}
	}
	
	if (arg_flags & ARG_SUPERBLOCK) {
		print_superblock(filesystem.superblock);
	}
	
	if (arg_flags & ARG_BLOCK_GROUPS) {
		print_block_groups(&filesystem);
	}
	
	if (arg_flags & ARG_INODE) {
		print_inode_by_number(&filesystem, inode);
	}

	ext2_filesystem_fini(&filesystem);

	return 0;
}


static void syntax_print(void)
{
	printf("syntax: ext2info --strict-check --superblock --block-groups --inode <i-number> <device_name>\n");
}

static void print_superblock(ext2_superblock_t *superblock)
{
	uint16_t magic;
	uint32_t first_block;
	uint32_t block_size;
	uint32_t fragment_size;
	uint32_t blocks_per_group;
	uint32_t fragments_per_group;
	uint32_t rev_major;
	uint16_t rev_minor;
	uint16_t state;
	uint32_t first_inode;
	uint16_t inode_size;
	uint32_t total_blocks;
	uint32_t reserved_blocks;
	uint32_t free_blocks;
	uint32_t total_inodes;
	uint32_t free_inodes;	
	uint32_t os;
	
	int pos;
	unsigned char c;
	
	magic = ext2_superblock_get_magic(superblock);
	first_block = ext2_superblock_get_first_block(superblock);
	block_size = ext2_superblock_get_block_size(superblock);
	fragment_size = ext2_superblock_get_fragment_size(superblock);
	blocks_per_group = ext2_superblock_get_blocks_per_group(superblock);
	fragments_per_group = ext2_superblock_get_fragments_per_group(superblock);
	rev_major = ext2_superblock_get_rev_major(superblock);
	rev_minor = ext2_superblock_get_rev_minor(superblock);
	state = ext2_superblock_get_state(superblock);
	first_inode = ext2_superblock_get_first_inode(superblock);
	inode_size = ext2_superblock_get_inode_size(superblock);
	total_blocks = ext2_superblock_get_total_block_count(superblock);
	reserved_blocks = ext2_superblock_get_reserved_block_count(superblock);
	free_blocks = ext2_superblock_get_free_block_count(superblock);
	total_inodes = ext2_superblock_get_total_inode_count(superblock);
	free_inodes = ext2_superblock_get_free_inode_count(superblock);
	os = ext2_superblock_get_os(superblock);
	
	printf("Superblock:\n");
	
	if (magic == EXT2_SUPERBLOCK_MAGIC) {
		printf("  Magic value: %X (correct)\n", magic);
	}
	else {
		printf("  Magic value: %X (incorrect)\n", magic);
	}
	
	printf("  Revision: %u.%hu\n", rev_major, rev_minor);
	printf("  State: %hu\n", state);
	printf("  Creator OS: %u\n", os);
	printf("  First block: %u\n", first_block);
	printf("  Block size: %u bytes (%u KiB)\n", block_size, block_size/1024);
	printf("  Blocks per group: %u\n", blocks_per_group);
	printf("  Total blocks: %u\n", total_blocks);
	printf("  Reserved blocks: %u\n", reserved_blocks);
	printf("  Free blocks: %u\n", free_blocks);
	printf("  Fragment size: %u bytes (%u KiB)\n", fragment_size,
	    fragment_size/1024);
	printf("  Fragments per group: %u\n", fragments_per_group);
	printf("  First inode: %u\n", first_inode);
	printf("  Inode size: %hu bytes\n", inode_size);
	printf("  Total inodes: %u\n", total_inodes);
	printf("  Free inodes: %u\n", free_inodes);
	
	
	if (rev_major == 1) {
		printf("  UUID: ");
		for (pos = 0; pos < 16; pos++) {
			printf("%02x", superblock->uuid[pos]);
		}
		printf("\n");
		
		printf("  Volume label: ");
		for (pos = 0; pos < 16; pos++) {
			c = superblock->volume_name[pos];
			if (c >= 32 && c < 128) {
				putchar(c);
			}
			else {
				putchar(' ');
			}
		}
		printf("\n");
	}
	
}

void print_block_groups(ext2_filesystem_t *filesystem)
{
	uint32_t block_group_count;
	uint32_t i;
	ext2_block_group_ref_t *block_group_ref;
	int rc;
	
	printf("Block groups:\n");
	
	block_group_count = ext2_superblock_get_block_group_count(
	    filesystem->superblock);
	
	for (i = 0; i < block_group_count; i++) {
		printf("  Block group %u\n", i);
		rc = ext2_filesystem_get_block_group_ref(filesystem, i, &block_group_ref);
		if (rc != EOK) {
			printf("    Failed reading block group\n");
			continue;
		}
		
		print_block_group(block_group_ref->block_group);
		
		rc = ext2_filesystem_put_block_group_ref(block_group_ref);
		if (rc != EOK) {
			printf("    Failed freeing block group\n");
		}
	}
	
}

void print_block_group(ext2_block_group_t *bg)
{
	uint32_t block_bitmap_block;
	uint32_t inode_bitmap_block;
	uint32_t inode_table_first_block;
	uint16_t free_block_count;
	uint16_t free_inode_count;
	uint16_t directory_inode_count;
	
	block_bitmap_block = ext2_block_group_get_block_bitmap_block(bg);
	inode_bitmap_block = ext2_block_group_get_inode_bitmap_block(bg);
	inode_table_first_block = ext2_block_group_get_inode_table_first_block(bg);
	free_block_count = ext2_block_group_get_free_block_count(bg);
	free_inode_count = ext2_block_group_get_free_inode_count(bg);
	directory_inode_count = ext2_block_group_get_directory_inode_count(bg);
	
	printf("    Block bitmap block: %u\n", block_bitmap_block);
	printf("    Inode bitmap block: %u\n", inode_bitmap_block);
	printf("    Inode table's first block: %u\n", inode_table_first_block);
	printf("    Free blocks: %u\n", free_block_count);
	printf("    Free inodes: %u\n", free_inode_count);
	printf("    Directory inodes: %u\n", directory_inode_count);
}

void print_inode_by_number(ext2_filesystem_t *fs, uint32_t inode)
{
	int rc;
	ext2_inode_ref_t *inode_ref;
	
	printf("Inode %u\n", inode);
	
	rc = ext2_filesystem_get_inode_ref(fs, inode, &inode_ref);
	if (rc != EOK) {
		printf("  Failed getting inode ref\n");
		return;
	}
	
	print_inode(inode_ref->inode);
	
	rc = ext2_filesystem_put_inode_ref(inode_ref);
	if (rc != EOK) {
		printf("  Failed putting inode ref\n");
	}
}

void print_inode(ext2_inode_t *inode)
{
	uint16_t mode;
	uint32_t user_id;
	uint32_t group_id;
	uint32_t size;
	uint16_t usage_count;
	uint32_t flags;
	uint16_t access;
	const char *type;
	uint32_t block;
	int i;
	bool all_blocks = false;
	
	mode = ext2_inode_get_mode(inode);
	user_id = ext2_inode_get_user_id(inode);
	group_id = ext2_inode_get_group_id(inode);
	size = ext2_inode_get_size(inode);
	usage_count = ext2_inode_get_usage_count(inode);
	flags = ext2_inode_get_flags(inode);
	
	type = "Unknown";
	if ((mode & EXT2_INODE_MODE_BLOCKDEV) == EXT2_INODE_MODE_BLOCKDEV) {
		type = "Block device";
	}
	else if (mode & EXT2_INODE_MODE_FIFO) {
		type = "Fifo (pipe)";
	}
	else if (mode & EXT2_INODE_MODE_CHARDEV) {
		type = "Character device";
	}
	else if (mode & EXT2_INODE_MODE_DIRECTORY) {
		type = "Directory";
	}
	else if (mode & EXT2_INODE_MODE_FILE) {
		type = "File";
	}
	else if (mode & EXT2_INODE_MODE_SOFTLINK) {
		type = "Soft link";
	}
	else if (mode & EXT2_INODE_MODE_SOCKET) {
		type = "Socket";
	}
	
	access = mode & EXT2_INODE_MODE_ACCESS_MASK;
	
	printf("  Mode: %04x (Type: %s, Access bits: %04ho)\n", mode, type, access);
	printf("  User ID: %u\n", user_id);
	printf("  Group ID: %u\n", group_id);
	printf("  Size: %u\n", size);
	printf("  Usage (link) count: %u\n", usage_count);
	printf("  Flags: %u\n", flags);
	printf("  Block list: ");
	for (i = 0; i < 12; i++) {
		block = ext2_inode_get_direct_block(inode, i);
		if (block == 0) {
			all_blocks = true;
			break;
		}
		printf("%u ", block);
	}
	if (!all_blocks) {
		printf(" and more...");
	}
	printf("\n");
}

/**
 * @}
 */
