/*
 * Copyright (c) 2015 Jiri Svoboda
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * - Redistributions of source code must retain the above copyright
 *   notice, this list of conditions and the following disclaimer.
 * - Redistributions in binary form must reproduce the above copyright
 *   notice, this list of conditions and the following disclaimer in the
 *   documentation and/or other materials provided with the distribution.
 * - The name of the author may not be used to endorse or promote products
 *   derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/** @addtogroup volsrv
 * @{
 */
/**
 * @file Filesystem creation
 * @brief
 */

#include <errno.h>
#include <io/log.h>
#include <loc.h>
#include <stdarg.h>
#include <stdlib.h>
#include <str_error.h>
#include <task.h>
#include <types/vol.h>

#include "mkfs.h"

static int cmd_runl(const char *path, ...)
{
	va_list ap;
	const char *arg;
	int cnt = 0;

	va_start(ap, path);
	do {
		arg = va_arg(ap, const char *);
		cnt++;
	} while (arg != NULL);
	va_end(ap);

	log_msg(LOG_DEFAULT, LVL_NOTE, "Calling task_spawn()");

	va_start(ap, path);
	task_id_t id;
	task_wait_t wait;
	int rc = task_spawn(&id, &wait, path, cnt, ap);
	va_end(ap);

	log_msg(LOG_DEFAULT, LVL_NOTE, "task_spawn() -> rc = %d", rc);

	if (rc != EOK) {
		log_msg(LOG_DEFAULT, LVL_ERROR, "Error spawning %s (%s)",
		    path, str_error(rc));
		return rc;
	}

	log_msg(LOG_DEFAULT, LVL_NOTE, "task_spawn() -> id = %d", (int)id);

	if (!id) {
		log_msg(LOG_DEFAULT, LVL_ERROR, "Error spawning %s "
		    "(invalid task ID)", path);
		return EINVAL;
	}

	log_msg(LOG_DEFAULT, LVL_NOTE, "call task_wait()");

	task_exit_t texit;
	int retval;
	rc = task_wait(&wait, &texit, &retval);
	log_msg(LOG_DEFAULT, LVL_NOTE, "task_wait() -> rc = %d", rc);
	if (rc != EOK) {
		log_msg(LOG_DEFAULT, LVL_ERROR, "Error waiting for %s (%s)",
		    path, str_error(rc));
		return rc;
	}

	log_msg(LOG_DEFAULT, LVL_NOTE, "task_wait() -> texit = %d", (int)texit);
	if (texit != TASK_EXIT_NORMAL) {
		log_msg(LOG_DEFAULT, LVL_ERROR, "Command %s unexpectedly "
		    "terminated", path);
		return EINVAL;
	}

	log_msg(LOG_DEFAULT, LVL_NOTE, "task_wait() -> retval = %d", (int)retval);
	if (retval != 0) {
		log_msg(LOG_DEFAULT, LVL_ERROR, "Command %s returned non-zero "
		    "exit code %d)", path, retval);
	}

	return retval;
}


int volsrv_part_mkfs(service_id_t sid, vol_fstype_t fstype)
{
	const char *cmd;
	char *svc_name;
	int rc;

	cmd = NULL;
	switch (fstype) {
	case fs_exfat:
		cmd = "/app/mkexfat";
		break;
	case fs_fat:
		cmd = "/app/mkfat";
		break;
	case fs_minix:
		cmd = "/app/mkmfs";
		break;
	case fs_ext4:
		cmd = NULL;
		break;
	}

	if (cmd == NULL)
		return ENOTSUP;

	rc = loc_service_get_name(sid, &svc_name);
	if (rc != EOK)
		return rc;

	log_msg(LOG_DEFAULT, LVL_NOTE, "volsrv_part_mkfs: Run '%s %s'.",
	    cmd, svc_name);

	rc = cmd_runl(cmd, cmd, svc_name, NULL);
	free(svc_name);
	return rc;
}

/** @}
 */
