#!/usr/bin/env python

import sys
import json
import re

def usage():
	print("%s - Automated structure and offsets generator" % sys.argv[0])
	print("%s file.ag probe|generate struct.ag" % sys.argv[0])
	sys.exit()

def generate_includes(struct):
	code = ""
	for i in range(len(struct['includes'])):
		code = code + "#include %s\n" % struct['includes'][i]
	return code.strip()

def generate_struct(struct):
	code = "typedef struct %s {\n" % struct['name']
	for i in range(len(struct['members'])):
		member = struct['members'][i]
		code = code + "\t%s %s;\n" % (member['type'], member['name'])
	code = code + "} %s_t;" % struct['name']
	return code

def generate_probes(struct):
	code = ""
	for i in range(len(struct['members'])):
		member = struct['members'][i]
		code = code + ("\temit_constant(OFFSET_%s, offsetof(%s_t, %s));\n" % 
		    (member['name'].upper(), struct['name'], member['name']))
	return code

def probe(struct):
	name = struct['name']
	typename = struct['name'] + "_t"

	code = """
%s		

#define str(s) #s
#define emit_constant(n, v) \
    asm volatile ("EMITTED_CONSTANT " str(n) \" = %%0\" :: \"i\" (v))
#define offsetof(t, m) ((size_t) &(((t *) 0)->m))

%s

main()
{
%s
	emit_constant(%s_SIZE, sizeof(%s));
}
	""" % (generate_includes(struct), generate_struct(struct),
	    generate_probes(struct), name.upper(), typename)

	return code

def generate_defines(pairs):
	code = ""
	for pair in pairs:
		code = code + "#define %s %s\n" % (pair[0], pair[1])
	return code.strip()

def generate(struct, lines):
	code = """
/***************************************************************************** 
 * AUTO-GENERATED FILE, DO NOT EDIT!!!
 * Generated by: tools/autogen.py
 * Generated from: %s
 *****************************************************************************/

#ifndef AUTOGEN_%s_H
#define AUTOGEN_%s_H

#ifndef __ASM__
%s
#endif

%s

#ifndef __ASM__
%s
#endif

#endif
	""" % (sys.argv[2], struct['name'].upper(), struct['name'].upper(),
	    generate_includes(struct), generate_defines(lines),
	    generate_struct(struct))

	return code

def filter_pairs(lines):
	pattern = re.compile("^\tEMITTED_CONSTANT ([A-Z_]*) = \$([0-9]*)$");
	pairs = []
	for line in lines:
		res = pattern.match(line)
		if res == None:
			continue
		pairs = pairs + [res.group(1, 2)]
	return pairs
	

def run():
	if len(sys.argv) != 3:
		usage()

	with open(sys.argv[2], "rb") as fp:
		struct = json.load(fp)

	if sys.argv[1] == "probe":
		code = probe(struct)
		print(code)
	elif sys.argv[1] == "generate":
		lines = sys.stdin.readlines()
		pairs = filter_pairs(lines)
		code = generate(struct, pairs)
		print(code)
	else:
		usage()

run()

