/*
 * Copyright (c) 2018 CZ.NIC, z.s.p.o.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * - Redistributions of source code must retain the above copyright
 *   notice, this list of conditions and the following disclaimer.
 * - Redistributions in binary form must reproduce the above copyright
 *   notice, this list of conditions and the following disclaimer in the
 *   documentation and/or other materials provided with the distribution.
 * - The name of the author may not be used to endorse or promote products
 *   derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <pcut/pcut.h>
#include <math.h>
#include <inttypes.h>
#include <float.h>

PCUT_INIT;

PCUT_TEST_SUITE(rounding);

static inline uint32_t fint(float x)
{
	union {
		float f;
		uint32_t i;
	} u = { .f = x };
	return u.i;
}

static inline uint64_t dint(double x)
{
	union {
		double f;
		uint64_t i;
	} u = { .f = x };
	return u.i;
}

#define assert_float_equals(x, y) PCUT_ASSERT_EQUALS(fint(x), fint(y))
#define assert_double_equals(x, y) PCUT_ASSERT_EQUALS(dint(x), dint(y))

#define FLOAT_CASES 200
#define DOUBLE_CASES 0

static float float_arguments[FLOAT_CASES] = {
	HUGE_VALF,
	-HUGE_VALF,
	__builtin_nanf(""),
	-__builtin_nanf(""),
	__builtin_nanf("0xdeadbe"),
	-__builtin_nanf("0xdeadbe"),

	0x0.000000p0f,       /* zero */
	0x0.000002p-126f,    /* smallest denormal > 0 */
	0x1.000000p-126f,    /* smallest normal > 0 */
	0x1.fffffep-2f,      /* largest < 0.5 */
	0x1.000000p-1f,      /* 0.5 */
	0x1.000002p-1f,      /* smallest > 0.5 */
	0x1.fffffep-1f,      /* largest < 1 */
	0x1.000000p0f,       /* 1 */
	0x1.000002p0f,       /* smallest > 1 */
	0x1.7ffffep0f,       /* largest < 1.5 */
	0x1.800000p0f,       /* 1.5 */
	0x1.800002p0f,       /* smallest > 1.5 */
	0x1.fffffep0f,       /* largest < 2 */
	0x1.000000p1f,       /* 2 */
	0x1.000002p1f,       /* smallest > 2 */
	0x1.3ffffep1f,       /* largest < 2.5 */
	0x1.400000p1f,       /* 2.5 */
	0x1.400002p1f,       /* smallest > 2.5 */
	0x1.7ffffep1f,       /* largest < 3 */
	0x1.800000p1f,       /* 3 */
	0x1.800002p1f,       /* smallest > 3 */
	0x1.bffffep1f,       /* largest < 3.5 */
	0x1.c00000p1f,       /* 3.5 */
	0x1.c00002p1f,       /* smallest > 3.5 */
	0x1.fffffep1f,       /* largest < 4 */
	0x1.000000p2f,       /* 4 */
	0x1.000002p2f,       /* smallest > 4 */

	0x1.ffffe0p20f,      /* 2^21 - 2 */
	0x1.ffffe2p20f,      /* 2^21 - 1.875 */
	0x1.ffffe4p20f,      /* 2^21 - 1.75 */
	0x1.ffffe6p20f,      /* 2^21 - 1.625 */
	0x1.ffffe8p20f,      /* 2^21 - 1.5 */
	0x1.ffffeap20f,      /* 2^21 - 1.375 */
	0x1.ffffecp20f,      /* 2^21 - 1.25 */
	0x1.ffffeep20f,      /* 2^21 - 1.125 */
	0x1.fffff0p20f,      /* 2^21 - 1 */
	0x1.fffff2p20f,      /* 2^21 - 0.875 */
	0x1.fffff4p20f,      /* 2^21 - 0.75 */
	0x1.fffff6p20f,      /* 2^21 - 0.625 */
	0x1.fffff8p20f,      /* 2^21 - 0.5 */
	0x1.fffffap20f,      /* 2^21 - 0.375 */
	0x1.fffffcp20f,      /* 2^21 - 0.25 */
	0x1.fffffep20f,      /* 2^21 - 0.125 */
	0x1.000000p21f,      /* 2^21 */
	0x1.000002p21f,      /* 2^21 + 0.25 */
	0x1.000004p21f,      /* 2^21 + 0.5 */
	0x1.000006p21f,      /* 2^21 + 0.75 */
	0x1.000008p21f,      /* 2^21 + 1 */
	0x1.00000ap21f,      /* 2^21 + 1.25 */
	0x1.00000cp21f,      /* 2^21 + 1.5 */
	0x1.00000ep21f,      /* 2^21 + 1.75 */
	0x1.000010p21f,      /* 2^21 + 2 */

	0x1.fffff0p21f,      /* 2^22 - 2 */
	0x1.fffff2p21f,      /* 2^22 - 1.75 */
	0x1.fffff4p21f,      /* 2^22 - 1.5 */
	0x1.fffff6p21f,      /* 2^22 - 1.25 */
	0x1.fffff8p21f,      /* 2^22 - 1 */
	0x1.fffffap21f,      /* 2^22 - 0.75 */
	0x1.fffffcp21f,      /* 2^22 - 0.5 */
	0x1.fffffep21f,      /* 2^22 - 0.25 */
	0x1.000000p22f,      /* 2^22 */
	0x1.000002p22f,      /* 2^22 + 0.5 */
	0x1.000004p22f,      /* 2^22 + 1 */
	0x1.000006p22f,      /* 2^22 + 1.5 */
	0x1.000008p22f,      /* 2^22 + 2 */

	0x1.fffff0p22f,      /* 2^23 - 4 */
	0x1.fffff2p22f,      /* 2^23 - 3.5 */
	0x1.fffff4p22f,      /* 2^23 - 3 */
	0x1.fffff6p22f,      /* 2^23 - 2.5 */
	0x1.fffff8p22f,      /* 2^23 - 2 */
	0x1.fffffap22f,      /* 2^23 - 1.5 */
	0x1.fffffcp22f,      /* 2^23 - 1 */
	0x1.fffffep22f,      /* 2^23 - 0.5 */
	0x1.000000p23f,      /* 2^23 */
	0x1.000002p23f,      /* 2^23 + 1 */
	0x1.000004p23f,      /* 2^23 + 2 */
	0x1.000006p23f,      /* 2^23 + 3 */
	0x1.000008p23f,      /* 2^23 + 4 */

	0x1.fffff0p23f,      /* 2^24 - 8 */
	0x1.fffff2p23f,      /* 2^24 - 7 */
	0x1.fffff4p23f,      /* 2^24 - 6 */
	0x1.fffff6p23f,      /* 2^24 - 5 */
	0x1.fffff8p23f,      /* 2^24 - 4 */
	0x1.fffffap23f,      /* 2^24 - 3 */
	0x1.fffffcp23f,      /* 2^24 - 2 */
	0x1.fffffep23f,      /* 2^24 - 1 */
	0x1.000000p24f,      /* 2^24 */
	0x1.000002p24f,      /* 2^24 + 2 */
	0x1.000004p24f,      /* 2^24 + 4 */
	0x1.000006p24f,      /* 2^24 + 6 */
	0x1.000008p24f,      /* 2^24 + 8 */

	0x1.fffffep100f,     /* large integer */

	/* Same as above but negative */

	-0x0.000000p0f,       /* zero */
	-0x0.000002p-126f,    /* smallest denormal > 0 */
	-0x1.000000p-126f,    /* smallest normal > 0 */
	-0x1.fffffep-2f,      /* largest < 0.5 */
	-0x1.000000p-1f,      /* 0.5 */
	-0x1.000002p-1f,      /* smallest > 0.5 */
	-0x1.fffffep-1f,      /* largest < 1 */
	-0x1.000000p0f,       /* 1 */
	-0x1.000002p0f,       /* smallest > 1 */
	-0x1.7ffffep0f,       /* largest < 1.5 */
	-0x1.800000p0f,       /* 1.5 */
	-0x1.800002p0f,       /* smallest > 1.5 */
	-0x1.fffffep0f,       /* largest < 2 */
	-0x1.000000p1f,       /* 2 */
	-0x1.000002p1f,       /* smallest > 2 */
	-0x1.3ffffep1f,       /* largest < 2.5 */
	-0x1.400000p1f,       /* 2.5 */
	-0x1.400002p1f,       /* smallest > 2.5 */
	-0x1.7ffffep1f,       /* largest < 3 */
	-0x1.800000p1f,       /* 3 */
	-0x1.800002p1f,       /* smallest > 3 */
	-0x1.bffffep1f,       /* largest < 3.5 */
	-0x1.c00000p1f,       /* 3.5 */
	-0x1.c00002p1f,       /* smallest > 3.5 */
	-0x1.fffffep1f,       /* largest < 4 */
	-0x1.000000p2f,       /* 4 */
	-0x1.000002p2f,       /* smallest > 4 */

	-0x1.ffffe0p20f,      /* 2^21 - 2 */
	-0x1.ffffe2p20f,      /* 2^21 - 1.875 */
	-0x1.ffffe4p20f,      /* 2^21 - 1.75 */
	-0x1.ffffe6p20f,      /* 2^21 - 1.625 */
	-0x1.ffffe8p20f,      /* 2^21 - 1.5 */
	-0x1.ffffeap20f,      /* 2^21 - 1.375 */
	-0x1.ffffecp20f,      /* 2^21 - 1.25 */
	-0x1.ffffeep20f,      /* 2^21 - 1.125 */
	-0x1.fffff0p20f,      /* 2^21 - 1 */
	-0x1.fffff2p20f,      /* 2^21 - 0.875 */
	-0x1.fffff4p20f,      /* 2^21 - 0.75 */
	-0x1.fffff6p20f,      /* 2^21 - 0.625 */
	-0x1.fffff8p20f,      /* 2^21 - 0.5 */
	-0x1.fffffap20f,      /* 2^21 - 0.375 */
	-0x1.fffffcp20f,      /* 2^21 - 0.25 */
	-0x1.fffffep20f,      /* 2^21 - 0.125 */
	-0x1.000000p21f,      /* 2^21 */
	-0x1.000002p21f,      /* 2^21 + 0.25 */
	-0x1.000004p21f,      /* 2^21 + 0.5 */
	-0x1.000006p21f,      /* 2^21 + 0.75 */
	-0x1.000008p21f,      /* 2^21 + 1 */
	-0x1.00000ap21f,      /* 2^21 + 1.25 */
	-0x1.00000cp21f,      /* 2^21 + 1.5 */
	-0x1.00000ep21f,      /* 2^21 + 1.75 */
	-0x1.000010p21f,      /* 2^21 + 2 */

	-0x1.fffff0p21f,      /* 2^22 - 2 */
	-0x1.fffff2p21f,      /* 2^22 - 1.75 */
	-0x1.fffff4p21f,      /* 2^22 - 1.5 */
	-0x1.fffff6p21f,      /* 2^22 - 1.25 */
	-0x1.fffff8p21f,      /* 2^22 - 1 */
	-0x1.fffffap21f,      /* 2^22 - 0.75 */
	-0x1.fffffcp21f,      /* 2^22 - 0.5 */
	-0x1.fffffep21f,      /* 2^22 - 0.25 */
	-0x1.000000p22f,      /* 2^22 */
	-0x1.000002p22f,      /* 2^22 + 0.5 */
	-0x1.000004p22f,      /* 2^22 + 1 */
	-0x1.000006p22f,      /* 2^22 + 1.5 */
	-0x1.000008p22f,      /* 2^22 + 2 */

	-0x1.fffff0p22f,      /* 2^23 - 4 */
	-0x1.fffff2p22f,      /* 2^23 - 3.5 */
	-0x1.fffff4p22f,      /* 2^23 - 3 */
	-0x1.fffff6p22f,      /* 2^23 - 2.5 */
	-0x1.fffff8p22f,      /* 2^23 - 2 */
	-0x1.fffffap22f,      /* 2^23 - 1.5 */
	-0x1.fffffcp22f,      /* 2^23 - 1 */
	-0x1.fffffep22f,      /* 2^23 - 0.5 */
	-0x1.000000p23f,      /* 2^23 */
	-0x1.000002p23f,      /* 2^23 + 1 */
	-0x1.000004p23f,      /* 2^23 + 2 */
	-0x1.000006p23f,      /* 2^23 + 3 */
	-0x1.000008p23f,      /* 2^23 + 4 */

	-0x1.fffff0p23f,      /* 2^24 - 8 */
	-0x1.fffff2p23f,      /* 2^24 - 7 */
	-0x1.fffff4p23f,      /* 2^24 - 6 */
	-0x1.fffff6p23f,      /* 2^24 - 5 */
	-0x1.fffff8p23f,      /* 2^24 - 4 */
	-0x1.fffffap23f,      /* 2^24 - 3 */
	-0x1.fffffcp23f,      /* 2^24 - 2 */
	-0x1.fffffep23f,      /* 2^24 - 1 */
	-0x1.000000p24f,      /* 2^24 */
	-0x1.000002p24f,      /* 2^24 + 2 */
	-0x1.000004p24f,      /* 2^24 + 4 */
	-0x1.000006p24f,      /* 2^24 + 6 */
	-0x1.000008p24f,      /* 2^24 + 8 */

	-0x1.fffffep100f,     /* large integer with full mantissa */

	/* a few random numbers*/
	3.5, -2.1, 100.0, 50.0, -1024.0, 0.0, 768.3156, 1080.499999, -600.0, 1.0
};

static float float_identity[FLOAT_CASES] = {
	HUGE_VALF,
	-HUGE_VALF,
	__builtin_nanf(""),
	-__builtin_nanf(""),
	__builtin_nanf("0xdeadbe"),
	-__builtin_nanf("0xdeadbe"),

	0.0,
	FLT_TRUE_MIN,    /* smallest denormal > 0 */
	FLT_MIN,         /* smallest normal > 0 */
	0.5 - (FLT_EPSILON / 4.),
	0.5,
	0.5 + (FLT_EPSILON / 2.),
	1.0 - (FLT_EPSILON / 2.),
	1.0,
	1.0 + FLT_EPSILON,
	1.5 - FLT_EPSILON,
	1.5,
	1.5 + FLT_EPSILON,
	2.0 - FLT_EPSILON,
	2.0,
	2.0 + (2.0 * FLT_EPSILON),
	2.5 - (2.0 * FLT_EPSILON),
	2.5,
	2.5 + (2.0 * FLT_EPSILON),
	3.0 - (2.0 * FLT_EPSILON),
	3.0,
	3.0 + (2.0 * FLT_EPSILON),
	3.5 - (2.0 * FLT_EPSILON),
	3.5,
	3.5 + (2.0 * FLT_EPSILON),
	4.0 - (2.0 * FLT_EPSILON),
	4.0,
	4.0 + (4.0 * FLT_EPSILON),

	2097150.000,     /* 2^21 - 2 */
	2097150.125,     /* 2^21 - 1.875 */
	2097150.250,     /* 2^21 - 1.75 */
	2097150.375,     /* 2^21 - 1.625 */
	2097150.500,     /* 2^21 - 1.5 */
	2097150.625,     /* 2^21 - 1.375 */
	2097150.750,     /* 2^21 - 1.25 */
	2097150.875,     /* 2^21 - 1.125 */
	2097151.000,     /* 2^21 - 1 */
	2097151.125,     /* 2^21 - 0.875 */
	2097151.250,     /* 2^21 - 0.75 */
	2097151.375,     /* 2^21 - 0.625 */
	2097151.500,     /* 2^21 - 0.5 */
	2097151.625,     /* 2^21 - 0.375 */
	2097151.750,     /* 2^21 - 0.25 */
	2097151.875,     /* 2^21 - 0.125 */
	2097152.00,      /* 2^21 */
	2097152.25,      /* 2^21 + 0.25 */
	2097152.50,      /* 2^21 + 0.5 */
	2097152.75,      /* 2^21 + 0.75 */
	2097153.00,      /* 2^21 + 1 */
	2097153.25,      /* 2^21 + 1.25 */
	2097153.50,      /* 2^21 + 1.5 */
	2097153.75,      /* 2^21 + 1.75 */
	2097154.00,      /* 2^21 + 2 */

	4194302.00,      /* 2^22 - 2 */
	4194302.25,      /* 2^22 - 1.75 */
	4194302.50,      /* 2^22 - 1.5 */
	4194302.75,      /* 2^22 - 1.25 */
	4194303.00,      /* 2^22 - 1 */
	4194303.25,      /* 2^22 - 0.75 */
	4194303.50,      /* 2^22 - 0.5 */
	4194303.75,      /* 2^22 - 0.25 */
	4194304.0,       /* 2^22 */
	4194304.5,       /* 2^22 + 0.5 */
	4194305.0,       /* 2^22 + 1 */
	4194305.5,       /* 2^22 + 1.5 */
	4194306.0,       /* 2^22 + 2 */

	8388604.0,      /* 2^23 - 4 */
	8388604.5,      /* 2^23 - 3.5 */
	8388605.0,      /* 2^23 - 3 */
	8388605.5,      /* 2^23 - 2.5 */
	8388606.0,      /* 2^23 - 2 */
	8388606.5,      /* 2^23 - 1.5 */
	8388607.0,      /* 2^23 - 1 */
	8388607.5,      /* 2^23 - 0.5 */
	8388608.0,      /* 2^23 */
	8388609.0,      /* 2^23 + 1 */
	8388610.0,      /* 2^23 + 2 */
	8388611.0,      /* 2^23 + 3 */
	8388612.0,      /* 2^23 + 4 */

	16777208.0,      /* 2^24 - 8 */
	16777209.0,      /* 2^24 - 7 */
	16777210.0,      /* 2^24 - 6 */
	16777211.0,      /* 2^24 - 5 */
	16777212.0,      /* 2^24 - 4 */
	16777213.0,      /* 2^24 - 3 */
	16777214.0,      /* 2^24 - 2 */
	16777215.0,      /* 2^24 - 1 */
	16777216.0,      /* 2^24 */
	16777218.0,      /* 2^24 + 2 */
	16777220.0,      /* 2^24 + 4 */
	16777222.0,      /* 2^24 + 6 */
	16777224.0,      /* 2^24 + 8 */

	0x1.fffffep100f,   /* large integer with full mantissa */

	/* Same as above but negative */

	-0.0,
	-FLT_TRUE_MIN,    /* smallest denormal > 0 */
	-FLT_MIN,         /* smallest normal > 0 */
	-(0.5 - (FLT_EPSILON / 4.)),
	-0.5,
	-(0.5 + (FLT_EPSILON / 2.)),
	-(1.0 - (FLT_EPSILON / 2.)),
	-1.0,
	-(1.0 + FLT_EPSILON),
	-(1.5 - FLT_EPSILON),
	-1.5,
	-(1.5 + FLT_EPSILON),
	-(2.0 - FLT_EPSILON),
	-2.0,
	-(2.0 + (2.0 * FLT_EPSILON)),
	-(2.5 - (2.0 * FLT_EPSILON)),
	-2.5,
	-(2.5 + (2.0 * FLT_EPSILON)),
	-(3.0 - (2.0 * FLT_EPSILON)),
	-3.0,
	-(3.0 + (2.0 * FLT_EPSILON)),
	-(3.5 - (2.0 * FLT_EPSILON)),
	-3.5,
	-(3.5 + (2.0 * FLT_EPSILON)),
	-(4.0 - (2.0 * FLT_EPSILON)),
	-4.0,
	-(4.0 + (4.0 * FLT_EPSILON)),

	-2097150.000,     /* 2^21 - 2 */
	-2097150.125,     /* 2^21 - 1.875 */
	-2097150.250,     /* 2^21 - 1.75 */
	-2097150.375,     /* 2^21 - 1.625 */
	-2097150.500,     /* 2^21 - 1.5 */
	-2097150.625,     /* 2^21 - 1.375 */
	-2097150.750,     /* 2^21 - 1.25 */
	-2097150.875,     /* 2^21 - 1.125 */
	-2097151.000,     /* 2^21 - 1 */
	-2097151.125,     /* 2^21 - 0.875 */
	-2097151.250,     /* 2^21 - 0.75 */
	-2097151.375,     /* 2^21 - 0.625 */
	-2097151.500,     /* 2^21 - 0.5 */
	-2097151.625,     /* 2^21 - 0.375 */
	-2097151.750,     /* 2^21 - 0.25 */
	-2097151.875,     /* 2^21 - 0.125 */
	-2097152.00,      /* 2^21 */
	-2097152.25,      /* 2^21 + 0.25 */
	-2097152.50,      /* 2^21 + 0.5 */
	-2097152.75,      /* 2^21 + 0.75 */
	-2097153.00,      /* 2^21 + 1 */
	-2097153.25,      /* 2^21 + 1.25 */
	-2097153.50,      /* 2^21 + 1.5 */
	-2097153.75,      /* 2^21 + 1.75 */
	-2097154.00,      /* 2^21 + 2 */

	-4194302.00,      /* 2^22 - 2 */
	-4194302.25,      /* 2^22 - 1.75 */
	-4194302.50,      /* 2^22 - 1.5 */
	-4194302.75,      /* 2^22 - 1.25 */
	-4194303.00,      /* 2^22 - 1 */
	-4194303.25,      /* 2^22 - 0.75 */
	-4194303.50,      /* 2^22 - 0.5 */
	-4194303.75,      /* 2^22 - 0.25 */
	-4194304.0,       /* 2^22 */
	-4194304.5,       /* 2^22 + 0.5 */
	-4194305.0,       /* 2^22 + 1 */
	-4194305.5,       /* 2^22 + 1.5 */
	-4194306.0,       /* 2^22 + 2 */

	-8388604.0,      /* 2^23 - 4 */
	-8388604.5,      /* 2^23 - 3.5 */
	-8388605.0,      /* 2^23 - 3 */
	-8388605.5,      /* 2^23 - 2.5 */
	-8388606.0,      /* 2^23 - 2 */
	-8388606.5,      /* 2^23 - 1.5 */
	-8388607.0,      /* 2^23 - 1 */
	-8388607.5,      /* 2^23 - 0.5 */
	-8388608.0,      /* 2^23 */
	-8388609.0,      /* 2^23 + 1 */
	-8388610.0,      /* 2^23 + 2 */
	-8388611.0,      /* 2^23 + 3 */
	-8388612.0,      /* 2^23 + 4 */

	-16777208.0,      /* 2^24 - 8 */
	-16777209.0,      /* 2^24 - 7 */
	-16777210.0,      /* 2^24 - 6 */
	-16777211.0,      /* 2^24 - 5 */
	-16777212.0,      /* 2^24 - 4 */
	-16777213.0,      /* 2^24 - 3 */
	-16777214.0,      /* 2^24 - 2 */
	-16777215.0,      /* 2^24 - 1 */
	-16777216.0,      /* 2^24 */
	-16777218.0,      /* 2^24 + 2 */
	-16777220.0,      /* 2^24 + 4 */
	-16777222.0,      /* 2^24 + 6 */
	-16777224.0,      /* 2^24 + 8 */

	-0x1.fffffep100f,   /* large integer with full mantissa */

	/* a few random numbers*/
	3.5, -2.1, 100.0, 50.0, -1024.0, 0.0, 768.3156, 1080.5, -600.0, 1.0
};

static float float_results_trunc[FLOAT_CASES] = {
	HUGE_VALF,
	-HUGE_VALF,
	__builtin_nanf(""),
	-__builtin_nanf(""),
	__builtin_nanf("0xdeadbe"),
	-__builtin_nanf("0xdeadbe"),

	0.0,
	0.0,
	0.0,
	0.0,
	0.0,
	0.0,
	0.0,
	1.0,
	1.0,
	1.0,
	1.0,
	1.0,
	1.0,
	2.0,
	2.0,
	2.0,
	2.0,
	2.0,
	2.0,
	3.0,
	3.0,
	3.0,
	3.0,
	3.0,
	3.0,
	4.0,
	4.0,

	2097150.000,     /* 2^21 - 2 */
	2097150.000,     /* 2^21 - 1.875 */
	2097150.000,     /* 2^21 - 1.75 */
	2097150.000,     /* 2^21 - 1.625 */
	2097150.000,     /* 2^21 - 1.5 */
	2097150.000,     /* 2^21 - 1.375 */
	2097150.000,     /* 2^21 - 1.25 */
	2097150.000,     /* 2^21 - 1.125 */
	2097151.000,     /* 2^21 - 1 */
	2097151.000,     /* 2^21 - 0.875 */
	2097151.000,     /* 2^21 - 0.75 */
	2097151.000,     /* 2^21 - 0.625 */
	2097151.000,     /* 2^21 - 0.5 */
	2097151.000,     /* 2^21 - 0.375 */
	2097151.000,     /* 2^21 - 0.25 */
	2097151.000,     /* 2^21 - 0.125 */
	2097152.00,      /* 2^21 */
	2097152.00,      /* 2^21 + 0.25 */
	2097152.00,      /* 2^21 + 0.5 */
	2097152.00,      /* 2^21 + 0.75 */
	2097153.00,      /* 2^21 + 1 */
	2097153.00,      /* 2^21 + 1.25 */
	2097153.00,      /* 2^21 + 1.5 */
	2097153.00,      /* 2^21 + 1.75 */
	2097154.00,      /* 2^21 + 2 */

	4194302.00,      /* 2^22 - 2 */
	4194302.00,      /* 2^22 - 1.75 */
	4194302.00,      /* 2^22 - 1.5 */
	4194302.00,      /* 2^22 - 1.25 */
	4194303.00,      /* 2^22 - 1 */
	4194303.00,      /* 2^22 - 0.75 */
	4194303.00,      /* 2^22 - 0.5 */
	4194303.00,      /* 2^22 - 0.25 */
	4194304.0,       /* 2^22 */
	4194304.0,       /* 2^22 + 0.5 */
	4194305.0,       /* 2^22 + 1 */
	4194305.0,       /* 2^22 + 1.5 */
	4194306.0,       /* 2^22 + 2 */

	8388604.0,      /* 2^23 - 4 */
	8388604.0,      /* 2^23 - 3.5 */
	8388605.0,      /* 2^23 - 3 */
	8388605.0,      /* 2^23 - 2.5 */
	8388606.0,      /* 2^23 - 2 */
	8388606.0,      /* 2^23 - 1.5 */
	8388607.0,      /* 2^23 - 1 */
	8388607.0,      /* 2^23 - 0.5 */
	8388608.0,      /* 2^23 */
	8388609.0,      /* 2^23 + 1 */
	8388610.0,      /* 2^23 + 2 */
	8388611.0,      /* 2^23 + 3 */
	8388612.0,      /* 2^23 + 4 */

	16777208.0,      /* 2^24 - 8 */
	16777209.0,      /* 2^24 - 7 */
	16777210.0,      /* 2^24 - 6 */
	16777211.0,      /* 2^24 - 5 */
	16777212.0,      /* 2^24 - 4 */
	16777213.0,      /* 2^24 - 3 */
	16777214.0,      /* 2^24 - 2 */
	16777215.0,      /* 2^24 - 1 */
	16777216.0,      /* 2^24 */
	16777218.0,      /* 2^24 + 2 */
	16777220.0,      /* 2^24 + 4 */
	16777222.0,      /* 2^24 + 6 */
	16777224.0,      /* 2^24 + 8 */

	0x1.fffffep100f,   /* large integer with full mantissa */

	/* Same as above but negative */

	-0.0,
	-0.0,
	-0.0,
	-0.0,
	-0.0,
	-0.0,
	-0.0,
	-1.0,
	-1.0,
	-1.0,
	-1.0,
	-1.0,
	-1.0,
	-2.0,
	-2.0,
	-2.0,
	-2.0,
	-2.0,
	-2.0,
	-3.0,
	-3.0,
	-3.0,
	-3.0,
	-3.0,
	-3.0,
	-4.0,
	-4.0,

	-2097150.000,     /* 2^21 - 2 */
	-2097150.000,     /* 2^21 - 1.875 */
	-2097150.000,     /* 2^21 - 1.75 */
	-2097150.000,     /* 2^21 - 1.625 */
	-2097150.000,     /* 2^21 - 1.5 */
	-2097150.000,     /* 2^21 - 1.375 */
	-2097150.000,     /* 2^21 - 1.25 */
	-2097150.000,     /* 2^21 - 1.125 */
	-2097151.000,     /* 2^21 - 1 */
	-2097151.000,     /* 2^21 - 0.875 */
	-2097151.000,     /* 2^21 - 0.75 */
	-2097151.000,     /* 2^21 - 0.625 */
	-2097151.000,     /* 2^21 - 0.5 */
	-2097151.000,     /* 2^21 - 0.375 */
	-2097151.000,     /* 2^21 - 0.25 */
	-2097151.000,     /* 2^21 - 0.125 */
	-2097152.00,      /* 2^21 */
	-2097152.00,      /* 2^21 + 0.25 */
	-2097152.00,      /* 2^21 + 0.5 */
	-2097152.00,      /* 2^21 + 0.75 */
	-2097153.00,      /* 2^21 + 1 */
	-2097153.00,      /* 2^21 + 1.25 */
	-2097153.00,      /* 2^21 + 1.5 */
	-2097153.00,      /* 2^21 + 1.75 */
	-2097154.00,      /* 2^21 + 2 */

	-4194302.00,      /* 2^22 - 2 */
	-4194302.00,      /* 2^22 - 1.75 */
	-4194302.00,      /* 2^22 - 1.5 */
	-4194302.00,      /* 2^22 - 1.25 */
	-4194303.00,      /* 2^22 - 1 */
	-4194303.00,      /* 2^22 - 0.75 */
	-4194303.00,      /* 2^22 - 0.5 */
	-4194303.00,      /* 2^22 - 0.25 */
	-4194304.0,       /* 2^22 */
	-4194304.0,       /* 2^22 + 0.5 */
	-4194305.0,       /* 2^22 + 1 */
	-4194305.0,       /* 2^22 + 1.5 */
	-4194306.0,       /* 2^22 + 2 */

	-8388604.0,      /* 2^23 - 4 */
	-8388604.0,      /* 2^23 - 3.5 */
	-8388605.0,      /* 2^23 - 3 */
	-8388605.0,      /* 2^23 - 2.5 */
	-8388606.0,      /* 2^23 - 2 */
	-8388606.0,      /* 2^23 - 1.5 */
	-8388607.0,      /* 2^23 - 1 */
	-8388607.0,      /* 2^23 - 0.5 */
	-8388608.0,      /* 2^23 */
	-8388609.0,      /* 2^23 + 1 */
	-8388610.0,      /* 2^23 + 2 */
	-8388611.0,      /* 2^23 + 3 */
	-8388612.0,      /* 2^23 + 4 */

	-16777208.0,      /* 2^24 - 8 */
	-16777209.0,      /* 2^24 - 7 */
	-16777210.0,      /* 2^24 - 6 */
	-16777211.0,      /* 2^24 - 5 */
	-16777212.0,      /* 2^24 - 4 */
	-16777213.0,      /* 2^24 - 3 */
	-16777214.0,      /* 2^24 - 2 */
	-16777215.0,      /* 2^24 - 1 */
	-16777216.0,      /* 2^24 */
	-16777218.0,      /* 2^24 + 2 */
	-16777220.0,      /* 2^24 + 4 */
	-16777222.0,      /* 2^24 + 6 */
	-16777224.0,      /* 2^24 + 8 */

	-0x1.fffffep100f,   /* large integer with full mantissa */

	/* a few random numbers*/
	3.0, -2.0, 100.0, 50.0, -1024.0, 0.0, 768.0, 1080.0, -600.0, 1.0
};

static float float_results_round[FLOAT_CASES] = {
	HUGE_VALF,
	-HUGE_VALF,
	__builtin_nanf(""),
	-__builtin_nanf(""),
	__builtin_nanf("0xdeadbe"),
	-__builtin_nanf("0xdeadbe"),

	0.0,
	0.0,    /* smallest denormal > 0 */
	0.0,         /* smallest normal > 0 */
	0.0,
	1.0,
	1.0,
	1.0,
	1.0,
	1.0,
	1.0,
	2.0,
	2.0,
	2.0,
	2.0,
	2.0,
	2.0,
	3.0,
	3.0,
	3.0,
	3.0,
	3.0,
	3.0,
	4.0,
	4.0,
	4.0,
	4.0,
	4.0,

	2097150.000,     /* 2^21 - 2 */
	2097150.000,     /* 2^21 - 1.875 */
	2097150.000,     /* 2^21 - 1.75 */
	2097150.000,     /* 2^21 - 1.625 */
	2097151.000,     /* 2^21 - 1.5 */
	2097151.000,     /* 2^21 - 1.375 */
	2097151.000,     /* 2^21 - 1.25 */
	2097151.000,     /* 2^21 - 1.125 */
	2097151.000,     /* 2^21 - 1 */
	2097151.000,     /* 2^21 - 0.875 */
	2097151.000,     /* 2^21 - 0.75 */
	2097151.000,     /* 2^21 - 0.625 */
	2097152.000,     /* 2^21 - 0.5 */
	2097152.000,     /* 2^21 - 0.375 */
	2097152.000,     /* 2^21 - 0.25 */
	2097152.000,     /* 2^21 - 0.125 */
	2097152.00,      /* 2^21 */
	2097152.00,      /* 2^21 + 0.25 */
	2097153.00,      /* 2^21 + 0.5 */
	2097153.00,      /* 2^21 + 0.75 */
	2097153.00,      /* 2^21 + 1 */
	2097153.00,      /* 2^21 + 1.25 */
	2097154.00,      /* 2^21 + 1.5 */
	2097154.00,      /* 2^21 + 1.75 */
	2097154.00,      /* 2^21 + 2 */

	4194302.00,      /* 2^22 - 2 */
	4194302.00,      /* 2^22 - 1.75 */
	4194303.00,      /* 2^22 - 1.5 */
	4194303.00,      /* 2^22 - 1.25 */
	4194303.00,      /* 2^22 - 1 */
	4194303.00,      /* 2^22 - 0.75 */
	4194304.00,      /* 2^22 - 0.5 */
	4194304.00,      /* 2^22 - 0.25 */
	4194304.0,       /* 2^22 */
	4194305.0,       /* 2^22 + 0.5 */
	4194305.0,       /* 2^22 + 1 */
	4194306.0,       /* 2^22 + 1.5 */
	4194306.0,       /* 2^22 + 2 */

	8388604.0,      /* 2^23 - 4 */
	8388605.0,      /* 2^23 - 3.5 */
	8388605.0,      /* 2^23 - 3 */
	8388606.0,      /* 2^23 - 2.5 */
	8388606.0,      /* 2^23 - 2 */
	8388607.0,      /* 2^23 - 1.5 */
	8388607.0,      /* 2^23 - 1 */
	8388608.0,      /* 2^23 - 0.5 */
	8388608.0,      /* 2^23 */
	8388609.0,      /* 2^23 + 1 */
	8388610.0,      /* 2^23 + 2 */
	8388611.0,      /* 2^23 + 3 */
	8388612.0,      /* 2^23 + 4 */

	16777208.0,      /* 2^24 - 8 */
	16777209.0,      /* 2^24 - 7 */
	16777210.0,      /* 2^24 - 6 */
	16777211.0,      /* 2^24 - 5 */
	16777212.0,      /* 2^24 - 4 */
	16777213.0,      /* 2^24 - 3 */
	16777214.0,      /* 2^24 - 2 */
	16777215.0,      /* 2^24 - 1 */
	16777216.0,      /* 2^24 */
	16777218.0,      /* 2^24 + 2 */
	16777220.0,      /* 2^24 + 4 */
	16777222.0,      /* 2^24 + 6 */
	16777224.0,      /* 2^24 + 8 */

	0x1.fffffep100f,   /* large integer with full mantissa */

	/* Same as above but negative */

	-0.0,
	-0.0,    /* smallest denormal > 0 */
	-0.0,         /* smallest normal > 0 */
	-0.0,
	-1.0,
	-1.0,
	-1.0,
	-1.0,
	-1.0,
	-1.0,
	-2.0,
	-2.0,
	-2.0,
	-2.0,
	-2.0,
	-2.0,
	-3.0,
	-3.0,
	-3.0,
	-3.0,
	-3.0,
	-3.0,
	-4.0,
	-4.0,
	-4.0,
	-4.0,
	-4.0,

	-2097150.000,     /* 2^21 - 2 */
	-2097150.000,     /* 2^21 - 1.875 */
	-2097150.000,     /* 2^21 - 1.75 */
	-2097150.000,     /* 2^21 - 1.625 */
	-2097151.000,     /* 2^21 - 1.5 */
	-2097151.000,     /* 2^21 - 1.375 */
	-2097151.000,     /* 2^21 - 1.25 */
	-2097151.000,     /* 2^21 - 1.125 */
	-2097151.000,     /* 2^21 - 1 */
	-2097151.000,     /* 2^21 - 0.875 */
	-2097151.000,     /* 2^21 - 0.75 */
	-2097151.000,     /* 2^21 - 0.625 */
	-2097152.000,     /* 2^21 - 0.5 */
	-2097152.000,     /* 2^21 - 0.375 */
	-2097152.000,     /* 2^21 - 0.25 */
	-2097152.000,     /* 2^21 - 0.125 */
	-2097152.00,      /* 2^21 */
	-2097152.00,      /* 2^21 + 0.25 */
	-2097153.00,      /* 2^21 + 0.5 */
	-2097153.00,      /* 2^21 + 0.75 */
	-2097153.00,      /* 2^21 + 1 */
	-2097153.00,      /* 2^21 + 1.25 */
	-2097154.00,      /* 2^21 + 1.5 */
	-2097154.00,      /* 2^21 + 1.75 */
	-2097154.00,      /* 2^21 + 2 */

	-4194302.00,      /* 2^22 - 2 */
	-4194302.00,      /* 2^22 - 1.75 */
	-4194303.00,      /* 2^22 - 1.5 */
	-4194303.00,      /* 2^22 - 1.25 */
	-4194303.00,      /* 2^22 - 1 */
	-4194303.00,      /* 2^22 - 0.75 */
	-4194304.00,      /* 2^22 - 0.5 */
	-4194304.00,      /* 2^22 - 0.25 */
	-4194304.0,       /* 2^22 */
	-4194305.0,       /* 2^22 + 0.5 */
	-4194305.0,       /* 2^22 + 1 */
	-4194306.0,       /* 2^22 + 1.5 */
	-4194306.0,       /* 2^22 + 2 */

	-8388604.0,      /* 2^23 - 4 */
	-8388605.0,      /* 2^23 - 3.5 */
	-8388605.0,      /* 2^23 - 3 */
	-8388606.0,      /* 2^23 - 2.5 */
	-8388606.0,      /* 2^23 - 2 */
	-8388607.0,      /* 2^23 - 1.5 */
	-8388607.0,      /* 2^23 - 1 */
	-8388608.0,      /* 2^23 - 0.5 */
	-8388608.0,      /* 2^23 */
	-8388609.0,      /* 2^23 + 1 */
	-8388610.0,      /* 2^23 + 2 */
	-8388611.0,      /* 2^23 + 3 */
	-8388612.0,      /* 2^23 + 4 */

	-16777208.0,      /* 2^24 - 8 */
	-16777209.0,      /* 2^24 - 7 */
	-16777210.0,      /* 2^24 - 6 */
	-16777211.0,      /* 2^24 - 5 */
	-16777212.0,      /* 2^24 - 4 */
	-16777213.0,      /* 2^24 - 3 */
	-16777214.0,      /* 2^24 - 2 */
	-16777215.0,      /* 2^24 - 1 */
	-16777216.0,      /* 2^24 */
	-16777218.0,      /* 2^24 + 2 */
	-16777220.0,      /* 2^24 + 4 */
	-16777222.0,      /* 2^24 + 6 */
	-16777224.0,      /* 2^24 + 8 */

	-0x1.fffffep100f,   /* large integer with full mantissa */

	/* a few random numbers*/
	4.0, -2.0, 100.0, 50.0, -1024.0, 0.0, 768.0, 1081.0, -600.0, 1.0
};

static float float_results_nearbyint[FLOAT_CASES] = {
	HUGE_VALF,
	-HUGE_VALF,
	__builtin_nanf(""),
	-__builtin_nanf(""),
	__builtin_nanf("0xdeadbe"),
	-__builtin_nanf("0xdeadbe"),

	0.0,
	0.0,    /* smallest denormal > 0 */
	0.0,         /* smallest normal > 0 */
	0.0,
	0.0,
	1.0,
	1.0,
	1.0,
	1.0,
	1.0,
	2.0,
	2.0,
	2.0,
	2.0,
	2.0,
	2.0,
	2.0,
	3.0,
	3.0,
	3.0,
	3.0,
	3.0,
	4.0,
	4.0,
	4.0,
	4.0,
	4.0,

	2097150.000,     /* 2^21 - 2 */
	2097150.000,     /* 2^21 - 1.875 */
	2097150.000,     /* 2^21 - 1.75 */
	2097150.000,     /* 2^21 - 1.625 */
	2097150.000,     /* 2^21 - 1.5 */
	2097151.000,     /* 2^21 - 1.375 */
	2097151.000,     /* 2^21 - 1.25 */
	2097151.000,     /* 2^21 - 1.125 */
	2097151.000,     /* 2^21 - 1 */
	2097151.000,     /* 2^21 - 0.875 */
	2097151.000,     /* 2^21 - 0.75 */
	2097151.000,     /* 2^21 - 0.625 */
	2097152.000,     /* 2^21 - 0.5 */
	2097152.000,     /* 2^21 - 0.375 */
	2097152.000,     /* 2^21 - 0.25 */
	2097152.000,     /* 2^21 - 0.125 */
	2097152.00,      /* 2^21 */
	2097152.00,      /* 2^21 + 0.25 */
	2097152.00,      /* 2^21 + 0.5 */
	2097153.00,      /* 2^21 + 0.75 */
	2097153.00,      /* 2^21 + 1 */
	2097153.00,      /* 2^21 + 1.25 */
	2097154.00,      /* 2^21 + 1.5 */
	2097154.00,      /* 2^21 + 1.75 */
	2097154.00,      /* 2^21 + 2 */

	4194302.00,      /* 2^22 - 2 */
	4194302.00,      /* 2^22 - 1.75 */
	4194302.00,      /* 2^22 - 1.5 */
	4194303.00,      /* 2^22 - 1.25 */
	4194303.00,      /* 2^22 - 1 */
	4194303.00,      /* 2^22 - 0.75 */
	4194304.00,      /* 2^22 - 0.5 */
	4194304.0,      /* 2^22 - 0.25 */
	4194304.0,       /* 2^22 */
	4194304.0,       /* 2^22 + 0.5 */
	4194305.0,       /* 2^22 + 1 */
	4194306.0,       /* 2^22 + 1.5 */
	4194306.0,       /* 2^22 + 2 */

	8388604.0,      /* 2^23 - 4 */
	8388604.0,      /* 2^23 - 3.5 */
	8388605.0,      /* 2^23 - 3 */
	8388606.0,      /* 2^23 - 2.5 */
	8388606.0,      /* 2^23 - 2 */
	8388606.0,      /* 2^23 - 1.5 */
	8388607.0,      /* 2^23 - 1 */
	8388608.0,      /* 2^23 - 0.5 */
	8388608.0,      /* 2^23 */
	8388609.0,      /* 2^23 + 1 */
	8388610.0,      /* 2^23 + 2 */
	8388611.0,      /* 2^23 + 3 */
	8388612.0,      /* 2^23 + 4 */

	16777208.0,      /* 2^24 - 8 */
	16777209.0,      /* 2^24 - 7 */
	16777210.0,      /* 2^24 - 6 */
	16777211.0,      /* 2^24 - 5 */
	16777212.0,      /* 2^24 - 4 */
	16777213.0,      /* 2^24 - 3 */
	16777214.0,      /* 2^24 - 2 */
	16777215.0,      /* 2^24 - 1 */
	16777216.0,      /* 2^24 */
	16777218.0,      /* 2^24 + 2 */
	16777220.0,      /* 2^24 + 4 */
	16777222.0,      /* 2^24 + 6 */
	16777224.0,      /* 2^24 + 8 */

	0x1.fffffep100f,   /* large integer with full mantissa */

	/* Same as above but negative */

	-0.0,
	-0.0,    /* smallest denormal > 0 */
	-0.0,         /* smallest normal > 0 */
	-0.0,
	-0.0,
	-1.0,
	-1.0,
	-1.0,
	-1.0,
	-1.0,
	-2.0,
	-2.0,
	-2.0,
	-2.0,
	-2.0,
	-2.0,
	-2.0,
	-3.0,
	-3.0,
	-3.0,
	-3.0,
	-3.0,
	-4.0,
	-4.0,
	-4.0,
	-4.0,
	-4.0,

	-2097150.000,     /* 2^21 - 2 */
	-2097150.000,     /* 2^21 - 1.875 */
	-2097150.000,     /* 2^21 - 1.75 */
	-2097150.000,     /* 2^21 - 1.625 */
	-2097150.000,     /* 2^21 - 1.5 */
	-2097151.000,     /* 2^21 - 1.375 */
	-2097151.000,     /* 2^21 - 1.25 */
	-2097151.000,     /* 2^21 - 1.125 */
	-2097151.000,     /* 2^21 - 1 */
	-2097151.000,     /* 2^21 - 0.875 */
	-2097151.000,     /* 2^21 - 0.75 */
	-2097151.000,     /* 2^21 - 0.625 */
	-2097152.000,     /* 2^21 - 0.5 */
	-2097152.000,     /* 2^21 - 0.375 */
	-2097152.000,     /* 2^21 - 0.25 */
	-2097152.000,     /* 2^21 - 0.125 */
	-2097152.00,      /* 2^21 */
	-2097152.00,      /* 2^21 + 0.25 */
	-2097152.00,      /* 2^21 + 0.5 */
	-2097153.00,      /* 2^21 + 0.75 */
	-2097153.00,      /* 2^21 + 1 */
	-2097153.00,      /* 2^21 + 1.25 */
	-2097154.00,      /* 2^21 + 1.5 */
	-2097154.00,      /* 2^21 + 1.75 */
	-2097154.00,      /* 2^21 + 2 */

	-4194302.00,      /* 2^22 - 2 */
	-4194302.00,      /* 2^22 - 1.75 */
	-4194302.00,      /* 2^22 - 1.5 */
	-4194303.00,      /* 2^22 - 1.25 */
	-4194303.00,      /* 2^22 - 1 */
	-4194303.00,      /* 2^22 - 0.75 */
	-4194304.00,      /* 2^22 - 0.5 */
	-4194304.0,      /* 2^22 - 0.25 */
	-4194304.0,       /* 2^22 */
	-4194304.0,       /* 2^22 + 0.5 */
	-4194305.0,       /* 2^22 + 1 */
	-4194306.0,       /* 2^22 + 1.5 */
	-4194306.0,       /* 2^22 + 2 */

	-8388604.0,      /* 2^23 - 4 */
	-8388604.0,      /* 2^23 - 3.5 */
	-8388605.0,      /* 2^23 - 3 */
	-8388606.0,      /* 2^23 - 2.5 */
	-8388606.0,      /* 2^23 - 2 */
	-8388606.0,      /* 2^23 - 1.5 */
	-8388607.0,      /* 2^23 - 1 */
	-8388608.0,      /* 2^23 - 0.5 */
	-8388608.0,      /* 2^23 */
	-8388609.0,      /* 2^23 + 1 */
	-8388610.0,      /* 2^23 + 2 */
	-8388611.0,      /* 2^23 + 3 */
	-8388612.0,      /* 2^23 + 4 */

	-16777208.0,      /* 2^24 - 8 */
	-16777209.0,      /* 2^24 - 7 */
	-16777210.0,      /* 2^24 - 6 */
	-16777211.0,      /* 2^24 - 5 */
	-16777212.0,      /* 2^24 - 4 */
	-16777213.0,      /* 2^24 - 3 */
	-16777214.0,      /* 2^24 - 2 */
	-16777215.0,      /* 2^24 - 1 */
	-16777216.0,      /* 2^24 */
	-16777218.0,      /* 2^24 + 2 */
	-16777220.0,      /* 2^24 + 4 */
	-16777222.0,      /* 2^24 + 6 */
	-16777224.0,      /* 2^24 + 8 */

	-0x1.fffffep100f,   /* large integer with full mantissa */

	/* a few random numbers*/
	4.0, -2.0, 100.0, 50.0, -1024.0, 0.0, 768.0, 1080.0, -600.0, 1.0
};

PCUT_TEST(identity)
{
	for (int i = 0; i < FLOAT_CASES; i++) {
		uint32_t f1 = fint(float_arguments[i]);
		uint32_t f2 = fint(float_identity[i]);
		if (f1 != f2) {
			PCUT_ASSERTION_FAILED("case %d: 0x%08x != 0x%08x\n", i, f1, f2);
		}
	}
}

PCUT_TEST(truncf)
{
	for (int i = 0; i < FLOAT_CASES; i++) {
		uint32_t f1 = fint(truncf(float_arguments[i]));
		uint32_t f2 = fint(float_results_trunc[i]);
		if (f1 != f2) {
			PCUT_ASSERTION_FAILED("case %d: 0x%08x != 0x%08x\n", i, f1, f2);
		}
	}
}

PCUT_TEST(trunc)
{
	for (int i = 0; i < FLOAT_CASES; i++) {
		uint64_t f1 = dint(trunc(float_arguments[i]));
		uint64_t f2 = dint(float_results_trunc[i]);
		if (f1 != f2) {
			PCUT_ASSERTION_FAILED("case %d: 0x%016" PRIx64 " != 0x%016" PRIx64 "\n", i, f1, f2);
		}
	}

	// TODO: double test cases
}

PCUT_TEST(roundf)
{
	for (int i = 0; i < FLOAT_CASES; i++) {
		uint32_t f1 = fint(roundf(float_arguments[i]));
		uint32_t f2 = fint(float_results_round[i]);
		if (f1 != f2) {
			PCUT_ASSERTION_FAILED("case %d: 0x%08x != 0x%08x\n", i, f1, f2);
		}
	}
}

PCUT_TEST(round)
{
	for (int i = 0; i < FLOAT_CASES; i++) {
		uint64_t f1 = dint(round(float_arguments[i]));
		uint64_t f2 = dint(float_results_round[i]);
		if (f1 != f2) {
			PCUT_ASSERTION_FAILED("case %d: 0x%016" PRIx64 " != 0x%016" PRIx64 "\n", i, f1, f2);
		}
	}

	// TODO: double test cases
}

PCUT_TEST(nearbyintf)
{
	// FIXME: ensure default rounding mode

	for (int i = 0; i < FLOAT_CASES; i++) {
		uint32_t f1 = fint(nearbyintf(float_arguments[i]));
		uint32_t f2 = fint(float_results_nearbyint[i]);
		if (f1 != f2) {
			PCUT_ASSERTION_FAILED("case %d: 0x%08x != 0x%08x\n", i, f1, f2);
		}
	}
}

PCUT_TEST(nearbyint)
{
	// FIXME: ensure default rounding mode

	for (int i = 0; i < FLOAT_CASES; i++) {
		uint64_t f1 = dint(nearbyint(float_arguments[i]));
		uint64_t f2 = dint(float_results_nearbyint[i]);
		if (f1 != f2) {
			PCUT_ASSERTION_FAILED("case %d: 0x%016" PRIx64 " != 0x%016" PRIx64 "\n", i, f1, f2);
		}
	}

	// TODO: double test cases
}

PCUT_EXPORT(rounding);
