/*
 * Copyright (c) 2008 Jakub Jermar
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * - Redistributions of source code must retain the above copyright
 *   notice, this list of conditions and the following disclaimer.
 * - Redistributions in binary form must reproduce the above copyright
 *   notice, this list of conditions and the following disclaimer in the
 *   documentation and/or other materials provided with the distribution.
 * - The name of the author may not be used to endorse or promote products
 *   derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/** @addtogroup fs
 * @{
 */ 

/**
 * @file	exfat_idx.c
 * @brief	Layer for translating exFAT entities to VFS node indices.
 */

#include "exfat.h"
#include "../../vfs/vfs.h"
#include <errno.h>
#include <str.h>
#include <adt/hash_table.h>
#include <adt/list.h>
#include <assert.h>
#include <fibril_synch.h>
#include <malloc.h>

/** Each instance of this type describes one interval of freed VFS indices. */
typedef struct {
	link_t		link;
	fs_index_t	first;
	fs_index_t	last;
} freed_t;

/**
 * Each instance of this type describes state of all VFS indices that
 * are currently unused.
 */
typedef struct {
	link_t link;
	service_id_t service_id;

	/** Next unassigned index. */
	fs_index_t next;
	/** Number of remaining unassigned indices. */
	uint64_t remaining;

	/** Sorted list of intervals of freed indices. */
	list_t freed_list;
} unused_t;

/** Mutex protecting the list of unused structures. */
static FIBRIL_MUTEX_INITIALIZE(unused_lock);

/** List of unused structures. */
static LIST_INITIALIZE(unused_list);

static void unused_initialize(unused_t *u, service_id_t service_id)
{
	link_initialize(&u->link);
	u->service_id = service_id;
	u->next = 0;
	u->remaining = ((uint64_t)((fs_index_t)-1)) + 1;
	list_initialize(&u->freed_list);
}

static unused_t *unused_find(service_id_t service_id, bool lock)
{
	unused_t *u;

	if (lock)
		fibril_mutex_lock(&unused_lock);
	list_foreach(unused_list, l) {
		u = list_get_instance(l, unused_t, link);
		if (u->service_id == service_id) 
			return u;
	}

	if (lock)
		fibril_mutex_unlock(&unused_lock);
	return NULL;
}

/** Mutex protecting the up_hash and ui_hash. */
static FIBRIL_MUTEX_INITIALIZE(used_lock);

/**
 * Global hash table of all used exfat_idx_t structures.
 * The index structures are hashed by the service_id, parent node's first
 * cluster and index within the parent directory.
 */ 
static hash_table_t up_hash;

#define UPH_SID_KEY	0
#define UPH_PFC_KEY	1
#define UPH_PDI_KEY	2

static size_t pos_key_hash(unsigned long key[])
{
	/* Inspired by Effective Java, 2nd edition. */
	size_t hash = 17;
	
	hash = 31 * hash + key[UPH_PFC_KEY];
	hash = 31 * hash + key[UPH_PDI_KEY];
	hash = 31 * hash + key[UPH_SID_KEY];
	
	return hash;
}

static size_t pos_hash(const link_t *item)
{
	exfat_idx_t *fidx = list_get_instance(item, exfat_idx_t, uph_link);
	
	unsigned long pkey[] = {
		[UPH_SID_KEY] = fidx->service_id,
		[UPH_PFC_KEY] = fidx->pfc,
		[UPH_PDI_KEY] = fidx->pdi,
	};
	
	return pos_key_hash(pkey);
}

static bool pos_match(unsigned long key[], size_t keys, const link_t *item)
{
	service_id_t service_id = (service_id_t)key[UPH_SID_KEY];
	exfat_cluster_t pfc;
	unsigned pdi;
	exfat_idx_t *fidx = list_get_instance(item, exfat_idx_t, uph_link);

	switch (keys) {
	case 1:
		return (service_id == fidx->service_id);
	case 3:
		pfc = (exfat_cluster_t) key[UPH_PFC_KEY];
		pdi = (unsigned) key[UPH_PDI_KEY];
		return (service_id == fidx->service_id) && (pfc == fidx->pfc) &&
		    (pdi == fidx->pdi);
	default:
		assert((keys == 1) || (keys == 3));
	}

	return 0;
}

static hash_table_ops_t uph_ops = {
	.hash = pos_hash,
	.key_hash = pos_key_hash,
	.match = pos_match,
	.equal = 0,
	.remove_callback = 0,
};

/**
 * Global hash table of all used fat_idx_t structures.
 * The index structures are hashed by the service_id and index.
 */
static hash_table_t ui_hash;

#define UIH_SID_KEY	0
#define UIH_INDEX_KEY	1

static size_t idx_key_hash(unsigned long key[])
{
	service_id_t service_id = (service_id_t)key[UIH_SID_KEY];
	fs_index_t index = (fs_index_t)key[UIH_INDEX_KEY];

	/* 
	 * Compute a simple hash unlimited by specific table size as per: 
	 * Effective Java, 2nd edition.
	 */
	size_t hash = 17;
	hash = 31 * hash + (size_t)service_id;
	hash = 31 * hash + (size_t)index;
	return hash;
}

static size_t idx_hash(const link_t *item)
{
	exfat_idx_t *fidx = list_get_instance(item, exfat_idx_t, uih_link);
	
	unsigned long ikey[] = {
		[UIH_SID_KEY] = fidx->service_id,
		[UIH_INDEX_KEY] = fidx->index,
	};

	return idx_key_hash(ikey);
}

static bool idx_match(unsigned long key[], size_t keys, const link_t *item)
{
	service_id_t service_id = (service_id_t)key[UIH_SID_KEY];
	fs_index_t index;
	exfat_idx_t *fidx = list_get_instance(item, exfat_idx_t, uih_link);

	switch (keys) {
	case 1:
		return (service_id == fidx->service_id);
	case 2:
		index = (fs_index_t) key[UIH_INDEX_KEY];
		return (service_id == fidx->service_id) &&
		    (index == fidx->index);
	default:
		assert((keys == 1) || (keys == 2));
	}

	return 0;
}

static void idx_remove_callback(link_t *item)
{
	exfat_idx_t *fidx = list_get_instance(item, exfat_idx_t, uih_link);

	free(fidx);
}

static hash_table_ops_t uih_ops = {
	.hash = idx_hash,
	.key_hash = idx_key_hash,
	.match = idx_match,
	.equal = 0,
	.remove_callback = idx_remove_callback,
};

/** Allocate a VFS index which is not currently in use. */
static bool exfat_index_alloc(service_id_t service_id, fs_index_t *index)
{
	unused_t *u;
	
	assert(index);
	u = unused_find(service_id, true);
	if (!u)
		return false;	

	if (list_empty(&u->freed_list)) {
		if (u->remaining) { 
			/*
			 * There are no freed indices, allocate one directly
			 * from the counter.
			 */
			*index = u->next++;
			--u->remaining;
			fibril_mutex_unlock(&unused_lock);
			return true;
		}
	} else {
		/* There are some freed indices which we can reuse. */
		freed_t *f = list_get_instance(list_first(&u->freed_list),
		    freed_t, link);
		*index = f->first;
		if (f->first++ == f->last) {
			/* Destroy the interval. */
			list_remove(&f->link);
			free(f);
		}
		fibril_mutex_unlock(&unused_lock);
		return true;
	}
	/*
	 * We ran out of indices, which is extremely unlikely with FAT16, but
	 * theoretically still possible (e.g. too many open unlinked nodes or
	 * too many zero-sized nodes).
	 */
	fibril_mutex_unlock(&unused_lock);
	return false;
}

/** If possible, coalesce two intervals of freed indices. */
static void try_coalesce_intervals(link_t *l, link_t *r, link_t *cur)
{
	freed_t *fl = list_get_instance(l, freed_t, link);
	freed_t *fr = list_get_instance(r, freed_t, link);

	if (fl->last + 1 == fr->first) {
		if (cur == l) {
			fl->last = fr->last;
			list_remove(r);
			free(r);
		} else {
			fr->first = fl->first;
			list_remove(l);
			free(l);
		}
	}
}

/** Free a VFS index, which is no longer in use. */
static void exfat_index_free(service_id_t service_id, fs_index_t index)
{
	unused_t *u;

	u = unused_find(service_id, true);
	assert(u);

	if (u->next == index + 1) {
		/* The index can be returned directly to the counter. */
		u->next--;
		u->remaining++;
	} else {
		/*
		 * The index must be returned either to an existing freed
		 * interval or a new interval must be created.
		 */
		link_t *lnk;
		freed_t *n;
		for (lnk = u->freed_list.head.next; lnk != &u->freed_list.head;
		    lnk = lnk->next) {
			freed_t *f = list_get_instance(lnk, freed_t, link);
			if (f->first == index + 1) {
				f->first--;
				if (lnk->prev != &u->freed_list.head)
					try_coalesce_intervals(lnk->prev, lnk,
					    lnk);
				fibril_mutex_unlock(&unused_lock);
				return;
			}
			if (f->last == index - 1) {
				f->last++;
				if (lnk->next != &u->freed_list.head)
					try_coalesce_intervals(lnk, lnk->next,
					    lnk);
				fibril_mutex_unlock(&unused_lock);
				return;
			}
			if (index > f->first) {
				n = malloc(sizeof(freed_t));
				/* TODO: sleep until allocation succeeds */
				assert(n);
				link_initialize(&n->link);
				n->first = index;
				n->last = index;
				list_insert_before(&n->link, lnk);
				fibril_mutex_unlock(&unused_lock);
				return;
			}

		}
		/* The index will form the last interval. */
		n = malloc(sizeof(freed_t));
		/* TODO: sleep until allocation succeeds */
		assert(n);
		link_initialize(&n->link);
		n->first = index;
		n->last = index;
		list_append(&n->link, &u->freed_list);
	}
	fibril_mutex_unlock(&unused_lock);
}

static int exfat_idx_create(exfat_idx_t **fidxp, service_id_t service_id)
{
	exfat_idx_t *fidx;

	fidx = (exfat_idx_t *) malloc(sizeof(exfat_idx_t));
	if (!fidx) 
		return ENOMEM;
	if (!exfat_index_alloc(service_id, &fidx->index)) {
		free(fidx);
		return ENOSPC;
	}
		
	link_initialize(&fidx->uph_link);
	link_initialize(&fidx->uih_link);
	fibril_mutex_initialize(&fidx->lock);
	fidx->service_id = service_id;
	fidx->pfc = 0;	/* no parent yet */
	fidx->pdi = 0;
	fidx->nodep = NULL;

	*fidxp = fidx;
	return EOK;
}

int exfat_idx_get_new(exfat_idx_t **fidxp, service_id_t service_id)
{
	exfat_idx_t *fidx;
	int rc;

	fibril_mutex_lock(&used_lock);
	rc = exfat_idx_create(&fidx, service_id);
	if (rc != EOK) {
		fibril_mutex_unlock(&used_lock);
		return rc;
	}
		
	hash_table_insert(&ui_hash, &fidx->uih_link);
	fibril_mutex_lock(&fidx->lock);
	fibril_mutex_unlock(&used_lock);

	*fidxp = fidx;
	return EOK;
}

exfat_idx_t *
exfat_idx_get_by_pos(service_id_t service_id, exfat_cluster_t pfc, unsigned pdi)
{
	exfat_idx_t *fidx;
	link_t *l;
	unsigned long pkey[] = {
		[UPH_SID_KEY] = service_id,
		[UPH_PFC_KEY] = pfc,
		[UPH_PDI_KEY] = pdi,
	};

	fibril_mutex_lock(&used_lock);
	l = hash_table_find(&up_hash, pkey);
	if (l) {
		fidx = hash_table_get_instance(l, exfat_idx_t, uph_link);
	} else {
		int rc;

		rc = exfat_idx_create(&fidx, service_id);
		if (rc != EOK) {
			fibril_mutex_unlock(&used_lock);
			return NULL;
		}
		
		fidx->pfc = pfc;
		fidx->pdi = pdi;

		hash_table_insert(&up_hash, &fidx->uph_link);
		hash_table_insert(&ui_hash, &fidx->uih_link);
	}
	fibril_mutex_lock(&fidx->lock);
	fibril_mutex_unlock(&used_lock);

	return fidx;
}

void exfat_idx_hashin(exfat_idx_t *idx)
{
	fibril_mutex_lock(&used_lock);
	hash_table_insert(&up_hash, &idx->uph_link);
	fibril_mutex_unlock(&used_lock);
}

void exfat_idx_hashout(exfat_idx_t *idx)
{
	unsigned long pkey[] = {
		[UPH_SID_KEY] = idx->service_id,
		[UPH_PFC_KEY] = idx->pfc,
		[UPH_PDI_KEY] = idx->pdi,
	};

	fibril_mutex_lock(&used_lock);
	hash_table_remove(&up_hash, pkey, 3);
	fibril_mutex_unlock(&used_lock);
}

exfat_idx_t *
exfat_idx_get_by_index(service_id_t service_id, fs_index_t index)
{
	exfat_idx_t *fidx = NULL;
	link_t *l;
	unsigned long ikey[] = {
		[UIH_SID_KEY] = service_id,
		[UIH_INDEX_KEY] = index,
	};

	fibril_mutex_lock(&used_lock);
	l = hash_table_find(&ui_hash, ikey);
	if (l) {
		fidx = hash_table_get_instance(l, exfat_idx_t, uih_link);
		fibril_mutex_lock(&fidx->lock);
	}
	fibril_mutex_unlock(&used_lock);

	return fidx;
}

/** Destroy the index structure.
 *
 * @param idx		The index structure to be destroyed.
 */
void exfat_idx_destroy(exfat_idx_t *idx)
{
	unsigned long ikey[] = {
		[UIH_SID_KEY] = idx->service_id,
		[UIH_INDEX_KEY] = idx->index,
	};
	service_id_t service_id = idx->service_id;
	fs_index_t index = idx->index;

	/* TODO: assert(idx->pfc == FAT_CLST_RES0); */
	assert(idx->pfc == 0);

	fibril_mutex_lock(&used_lock);
	/*
	 * Since we can only free unlinked nodes, the index structure is not
	 * present in the position hash (uph). We therefore hash it out from
	 * the index hash only.
	 */
	hash_table_remove(&ui_hash, ikey, 2);
	fibril_mutex_unlock(&used_lock);
	/* Release the VFS index. */
	exfat_index_free(service_id, index);
	/* The index structure itself is freed in idx_remove_callback(). */
}

int exfat_idx_init(void)
{
	if (!hash_table_create(&up_hash, 0, 3, &uph_ops)) 
		return ENOMEM;
	if (!hash_table_create(&ui_hash, 0, 2, &uih_ops)) {
		hash_table_destroy(&up_hash);
		return ENOMEM;
	}
	return EOK;
}

void exfat_idx_fini(void)
{
	/* We assume the hash tables are empty. */
	hash_table_destroy(&up_hash);
	hash_table_destroy(&ui_hash);
}

int exfat_idx_init_by_service_id(service_id_t service_id)
{
	unused_t *u;
	int rc = EOK;

	u = (unused_t *) malloc(sizeof(unused_t));
	if (!u)
		return ENOMEM;
	unused_initialize(u, service_id);
	fibril_mutex_lock(&unused_lock);
	if (!unused_find(service_id, false)) {
		list_append(&u->link, &unused_list);
	} else {
		free(u);
		rc = EEXIST;
	}
	fibril_mutex_unlock(&unused_lock);
	return rc;
}

void exfat_idx_fini_by_service_id(service_id_t service_id)
{
	unsigned long ikey[] = {
		[UIH_SID_KEY] = service_id 
	};
	unsigned long pkey[] = {
		[UPH_SID_KEY] = service_id 
	};

	/*
	 * Remove this instance's index structure from up_hash and ui_hash.
	 * Process up_hash first and ui_hash second because the index structure
	 * is actually removed in idx_remove_callback(). 
	 */
	fibril_mutex_lock(&used_lock);
	hash_table_remove(&up_hash, pkey, 1);
	hash_table_remove(&ui_hash, ikey, 1);
	fibril_mutex_unlock(&used_lock);

	/*
	 * Free the unused and freed structures for this instance.
	 */
	unused_t *u = unused_find(service_id, true);
	assert(u);
	list_remove(&u->link);
	fibril_mutex_unlock(&unused_lock);

	while (!list_empty(&u->freed_list)) {
		freed_t *f;
		f = list_get_instance(list_first(&u->freed_list), freed_t, link);
		list_remove(&f->link);
		free(f);
	}
	free(u); 
}

/**
 * @}
 */ 
